<?php

declare(strict_types=1);

namespace Drupal\bankid;

use Drupal\bankid\Enum\OrderOperation;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\TempStore\PrivateTempStoreFactory;

/**
 * Provides mock BankID responses for testing and development.
 */
class BankIDMockClient implements BankIDClientInterface {

  private PrivateTempStore $mockStore;

  /**
   * Constructor for BankIDMockService.
   */
  public function __construct(
    PrivateTempStoreFactory $tempStoreFactory,
  ) {
    $this->mockStore = $tempStoreFactory->get('bankid_mock');
  }

  /**
   * Mock the start endpoint response.
   */
  public function start(OrderOperation $operation, $endUserIp = '127.0.0.1', $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse {
    $orderRef = 'mock-' . bin2hex(random_bytes(16));
    $mockData = [
      'orderRef' => $orderRef,
    ];

    $this->mockStore->set($orderRef, $mockData);

    $responseBody = [
      'orderRef' => $orderRef,
      'status' => BankIDResponse::STATUS_PENDING,
      'qrStartToken' => 'mock-token-' . bin2hex(random_bytes(8)),
      'qrStartSecret' => 'mock-secret-' . bin2hex(random_bytes(8)),
      'autoStartToken' => 'mock-auto-' . bin2hex(random_bytes(8)),
    ];

    return new BankIDResponse(BankIDResponse::STATUS_PENDING, $responseBody);
  }

  /**
   * Mock the collect endpoint response.
   */
  public function collect($orderReference): BankIDResponse {
    $mockData = $this->mockStore->get($orderReference);
    if (!isset($mockData) || $mockData['orderRef'] !== $orderReference) {
      return new BankIDResponse(BankIDResponse::STATUS_FAILED, [
        'orderRef' => $orderReference,
        'errorCode' => BankIDResponse::ERROR_CODE_NOT_FOUND,
        'details' => 'Order not found',
      ]);
    }

    $outcome = $this->getCollectOutcome();
    $this->deleteCollectOutcome();
    switch ($outcome) {
      case 'cancel':
        $this->mockStore->delete($orderReference);
        return new BankIDResponse(BankIDResponse::STATUS_FAILED, [
          'orderRef' => $orderReference,
          'hintCode' => BankIDResponse::HINT_CODE_USER_CANCEL,
        ]);
      case 'expired':
        $this->mockStore->delete($orderReference);
        return new BankIDResponse(BankIDResponse::STATUS_FAILED, [
          'orderRef' => $orderReference,
          'hintCode' => BankIDResponse::HINT_CODE_EXPIRED_TRANSACTION,
        ]);
      case 'certificate_error':
        $this->mockStore->delete($orderReference);
        return new BankIDResponse(BankIDResponse::STATUS_FAILED, [
          'orderRef' => $orderReference,
          'hintCode' => BankIDResponse::HINT_CODE_CERTIFICATE_ERROR,
        ]);
    }

    // Check if personal number has been set
    $mock_user = $this->getMockCompletionUser();
    if (!isset($mock_user['personalNumber'])) {
      // Still waiting for user to submit personal number
      return new BankIDResponse(BankIDResponse::STATUS_PENDING, [
        'orderRef' => $orderReference,
        'hintCode' => BankIDResponse::HINT_CODE_OUTSTANDING_TRANSACTION,
      ]);
    }

    if (!isset($mockData['hasCollectedOnce'])) {
      // First collect - return user sign
      $mockData['hasCollectedOnce'] = TRUE;
      $this->mockStore->set($orderReference, $mockData);

      return new BankIDResponse(BankIDResponse::STATUS_PENDING, [
        'orderRef' => $orderReference,
        'hintCode' => BankIDResponse::HINT_CODE_USER_SIGN,
      ]);
    }

    // Second collect - return complete
    $this->mockStore->delete($orderReference);
    $this->deleteMockCompletionUser();

    return new BankIDResponse(BankIDResponse::STATUS_COMPLETE, [
      'orderRef' => $orderReference,
      'completionData' => [
        'user' => $mock_user,
        'device' => [
          'ipAddress' => '192.0.2.1',
          'uhi' => 'Mock',
        ],
        'bankIdIssueDate' => date('Y-m-d'),
        // @todo Mock xml signature
        'signature' => base64_encode('mock-signature-' . bin2hex(random_bytes(32))),
        'ocspResponse' => base64_encode('mock-ocsp-' . bin2hex(random_bytes(32))),
      ],
    ]);
  }

  /**
   * Mock the cancel endpoint response.
   */
  public function cancel($orderReference): BankIDResponse {
    $this->mockStore->delete($orderReference);

    return new BankIDResponse(BankIDResponse::STATUS_OK, []);
  }

  public function setMockCompletionUser(string $personalNumber, string $givenName, string $surname): void {
    $user = [
      'personalNumber' => $personalNumber,
      'name' => $givenName . ' ' . $surname,
      'givenName' => $givenName,
      'surname' => $surname,
    ];
    $this->mockStore->set('completion-user', $user);
  }

  private function getMockCompletionUser() {
    return $this->mockStore->get('completion-user');
  }

  private function deleteMockCompletionUser(): void {
    $this->mockStore->delete('completion-user');
  }

  public function setCollectOutcome(string $outcome): void {
    $this->mockStore->set('collect-outcome', $outcome);
  }

  private function getCollectOutcome(): string {
    return $this->mockStore->get('collect-outcome') ?? 'none';
  }

  private function deleteCollectOutcome() {
    $this->mockStore->delete('collect-outcome');
  }

  public function phoneAuth($personalNumber, $callInitiator, $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse {
    return new BankIDResponse(BankIDResponse::STATUS_PENDING, []);
  }

  public function phoneSign($personalNumber, $callInitiator, $requirement = NULL, $userVisibleData = NULL, $userNonVisibleData = NULL, $userVisibleDataFormat = NULL): BankIDResponse {
    return new BankIDResponse(BankIDResponse::STATUS_PENDING, []);
  }

}
