<?php

declare(strict_types=1);

namespace Drupal\bankid;

use Drupal\bankid\Enum\OrderOperation;
use Drupal\bankid\ValueObjects\BankIDOrderDoneData;
use Drupal\bankid\ValueObjects\CaseData;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Manages BankID orders.
 *
 * @todo Refactor to be more aligned with SOLID principles, mainly smaller
 *   "single" responsibility classes/services, e.g BankIDTempStore, Move Plugin
 *   loading to BankIDCasePluginManager, etc.
 *   Add interface.
 */
final class BankIDOrderManager {

  use StringTranslationTrait;

  /**
   * The private temporary store.
   */
  private PrivateTempStore $privateTempStore;

  /**
   * The key value store.
   */
  private KeyValueStoreInterface $keyValueStore;

  /**
   * The BankID configuration.
   */
  private ImmutableConfig $config;

  /**
   * Constructor for BankIDOrderManager.
   */
  public function __construct(
    private readonly BankIDClientInterface $bankID,
    TranslationInterface $stringTranslation,
    PrivateTempStoreFactory $tempStoreFactory,
    KeyValueFactoryInterface $keyValueFactory,
    private readonly BankIDCasePluginManager $bankIDCaseManager,
    private readonly RequestStack $requestStack,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->stringTranslation = $stringTranslation;
    $this->privateTempStore = $tempStoreFactory->get('bankid');
    $this->keyValueStore = $keyValueFactory->get('bankid');
    $this->config = $config_factory->get('bankid.settings');
  }

  /**
   * Register a case.
   *
   * @param \Drupal\bankid\ValueObjects\CaseData $case
   *   The case to register.
   *
   * @return string
   *   The configuration hash of the registered case.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  public function registerCase(CaseData $case): string {
    $public_serialized = serialize($case);
    $public_hash = hash('sha256', $public_serialized);
    $this->keyValueStore->set($public_hash, $public_serialized);

    if (is_array($case->private) && count($case->private) > 0) {
      $private_serialized = serialize($case->private);
      $private_hash = hash('sha256', $private_serialized);
      $this->privateTempStore->set($private_hash, $private_serialized);
      return $public_hash . ':' . $private_hash;
    }

    return $public_hash;
  }

  /**
   * Load a plugin.
   *
   * @todo The methods are named self::registerCase() and self::loadPlugin().
   *   They seem to be related, should they be named self::registerCase() and
   *   self::loadCase() instead?
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \Drupal\bankid\BankIDException
   */
  public function loadPlugin(?string $configuration_hash): ?BankIDCaseInterface {
    $case = $this->getCaseData($configuration_hash);
    if ($case instanceof CaseData) {
      $plugin = $this->bankIDCaseManager->createInstance(
        $case->pluginId,
        ['operation' => $case->operation] + ($case->pluginConfiguration ?? [])
      );

      if ($plugin instanceof BankIDCaseInterface) {
        $plugin->setPrivateData($case->private ?? []);
      }

      return $plugin instanceof BankIDCaseInterface ? $plugin : NULL;
    }
    // @todo Replace default with setting.
    $plugin_id = 'default';
    $plugin = $this->bankIDCaseManager->createInstance($plugin_id, ['operation' => OrderOperation::Auth]);
    return $plugin instanceof BankIDCaseInterface ? $plugin : NULL;
  }

  /**
   * Get case data.
   *
   * @param string|null $configuration_hash
   *   The configuration hash to identify the configuration for this bankid
   *   order.
   *
   * @return \Drupal\bankid\ValueObjects\CaseData|null
   *   The case data or NULL.
   *
   * @throws \Drupal\bankid\BankIDException
   */
  public function getCaseData(?string $configuration_hash): ?CaseData {
    $hash_parts = explode(':', $configuration_hash ?? '');
    $public_hash = $hash_parts[0] ?? NULL;
    $private_hash = $hash_parts[1] ?? NULL;

    $case = NULL;
    $public_serialized = $this->keyValueStore->get($public_hash);
    if (is_string($public_serialized)) {
      $case = unserialize($public_serialized, ['allowed_classes' => [CaseData::class]]);
    }
    if (!($case instanceof CaseData)) {
      throw new BankIDException('Configuration not found.');
    }

    $private_serialized = is_string($private_hash) ? $this->privateTempStore->get($private_hash) : NULL;
    if (is_string($private_serialized)) {
      $private = unserialize($private_serialized, ['allowed_classes' => FALSE]);
      $case->setPrivate($private);
    }
    elseif (is_string($private_hash)) {
      throw new BankIDException('Private configuration not found.');
    }

    return $case;
  }

  /**
   * Start a sign order.
   *
   * @param string|null $configuration_hash
   *   The configuration hash to identify the configuration for this bankid
   *   order.
   *
   * @return array
   *   The result of the sign order start.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function startOrder(?string $configuration_hash = NULL): array {
    $case_data = $this->getCaseData($configuration_hash);
    $bankid_case = $this->loadPlugin($configuration_hash);

    $endUserIp = $this->requestStack->getCurrentRequest()->getClientIp();
    $result = $this->bankID->start(
      operation: $case_data->operation,
      endUserIp: $endUserIp,
      requirement: $bankid_case->getRequirements(),
      userVisibleData: $bankid_case->getVisibleData(),
      userNonVisibleData: $bankid_case->getHiddenData(),
      userVisibleDataFormat: $bankid_case->getVisibleDataFormat()
    );
    return $this->processOrderStart($result, $configuration_hash);
  }

  /**
   * Process the start of an order.
   *
   * @param \Drupal\bankid\BankIDResponse $result
   *   The BankID response object containing the order start data.
   * @param string $configuration_hash
   *   The configuration hash to identify the plugin configuration for this
   *   bankid order.
   *
   * @return array
   *   An array containing the order start data.
   */
  private function processOrderStart(
    BankIDResponse $result,
    string $configuration_hash,
  ): array {
    // @todo handle $result_data['status'] === 'failed (wrong cert).
    $result_data = $result->getBody();
    $order_ref = $result_data['orderRef'] ?? '';

    $this->setData($order_ref, 'configuration_hash', $configuration_hash);
    $this->setData($order_ref, 'qrStartSecret', $result_data['qrStartSecret']);
    $this->setData($order_ref, 'qrStartToken', $result_data['qrStartToken']);
    $this->setData($order_ref, 'startTime', time());

    unset($result_data['qrStartSecret']);

    return $result_data;
  }

  /**
   * Get the status of an order.
   *
   * @param string $order_ref
   *   The order reference to check the status for.
   *
   * @return array
   *   An array containing the status.
   */
  public function getOrderStatus(string $order_ref): array {
    $start_time = $this->getData($order_ref, 'startTime');
    $time = is_numeric($start_time) ? time() - $start_time : 0;

    if ($this->getData($order_ref, 'qrStartSecret') === NULL || $this->getData(
        $order_ref,
        'qrStartToken'
      ) === NULL) {
      return [
        'status' => BankIDResponse::STATUS_FAILED,
        'message' => $this->t(
          'QR code data is missing. Please start a new authentication.'
        ),
        'orderRef' => $order_ref,
        'qrData' => '',
        'hintCode' => NULL,
      ];
    }

    $lastFetchedData = $this->getData($order_ref, 'lastFetchedData');
    if (!($time % 2) || is_null($lastFetchedData)) {
      $data = $this->fetchCollect($order_ref);
      $this->setData($order_ref, 'lastFetchedData', $data);
    }
    else {
      $data = $this->getData($order_ref, 'lastFetchedData') ?? [];
    }

    if ($data['status'] === BankIDResponse::STATUS_PENDING) {
      $data['qrData'] = $this->buildQRData($order_ref, $time);
    }
    else {
      // Clean up everything not suffixed by 'only-expire'. This removes
      // 'secretstarttoken', 'qrstarttoken' and 'qrstartsecret' but keeps the
      // rest until it expires naturally.
      $this->deleteKey($order_ref);
    }

    return $data;
  }

  /**
   * Build the QR code data for the JS client.
   *
   * @param string $order_ref
   *   The order reference to build the QR code data for.
   * @param int $time
   *   The current time in seconds since the start of the order.
   *
   * @return string
   *   The formatted QR code data string.
   */
  // phpcs:ignore Drupal.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
  private function buildQRData(string $order_ref, $time): string {
    $qr_auth_code = hash_hmac(
      'sha256',
      (string) $time,
      $this->getData($order_ref, 'qrStartSecret')
    );
    $qr_start_token = $this->getData($order_ref, 'qrStartToken') ?? '';

    return "bankid.$qr_start_token.$time.$qr_auth_code";
  }

  /**
   * Fetch and parse collect data from BankID.
   *
   * @param string $order_ref
   *   The order reference.
   *
   * @return array
   *   An array containing the status, order reference, hint code, and message
   *   and optionally data from plugin if the order is complete.
   */
  private function fetchCollect($order_ref): array {
    $result = $this->bankID->collect($order_ref);
    $result_data = $result->getBody();

    $message = $this->getMessage($result_data['shortName'] ?? '');
    $status = $result_data['status'];

    $data = [
      'status' => $status,
      'orderRef' => $order_ref,
      'hintCode' => $result_data['hintCode'] ?? NULL,
    ];

    if (!empty($message)) {
      $data['message'] = $message;
    }

    $configuration_hash = $this->getData($order_ref, 'configuration_hash');

    $order_done_data = NULL;
    switch ($status) {
      case BankIDResponse::STATUS_PENDING:
        break;

      case BankIDResponse::STATUS_COMPLETE:
        $data['message'] = $this->t('Operation completed successfully.');
        $bankid_case = $this->loadPlugin($configuration_hash);
        $order_done_data = $bankid_case->orderCompleted($result_data);
        break;

      case BankIDResponse::STATUS_FAILED:
        $bankid_case = $this->loadPlugin($configuration_hash);
        $order_done_data = $bankid_case->orderFailed($result_data);
        // Print error message.
        break;

      case BankIDResponse::STATUS_OK:
        // Is canceled, modal is already closed.
        break;

      default:
        // Unknown status, print error message.
        $data['status'] = BankIDResponse::STATUS_FAILED;
        $data['message'] = $this->t('Unknown status received from BankID.');
        break;
    }

    if ($order_done_data instanceof BankIDOrderDoneData) {
      // Persist completion data for later use.
      $this->setData(
        $order_ref . ':only-expire',
        'uid',
        $order_done_data->getUserId()
      );
      $this->setData(
        $order_ref . ':only-expire',
        'orderDoneTempData',
        $order_done_data->getOrderDoneTempData()
      );

      // Data only for JS client.
      $data['data'] = $order_done_data->getClientData();
    }

    return $data;
  }

  /**
   * Cancel an order.
   *
   * @param string $order_ref
   *   The order reference.
   *
   * @return array
   *   An array containing the status, order reference, hint code, and message.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \Drupal\bankid\BankIDException
   */
  public function cancelOrder($order_ref): array {
    $configuration_hash = $this->getData($order_ref, 'configuration_hash');
    $bankid_case = $this->loadPlugin($configuration_hash);
    $response = $this->bankID->cancel($order_ref);
    $result_data = $response->getBody();
    $order_done_data = $bankid_case->orderCancelled($result_data, $order_ref);
    $this->setData(
      $order_ref . ':only-expire',
      'uid',
      $order_done_data->getUserId()
    );
    $this->setData(
      $order_ref . ':only-expire',
      'orderDoneTempData',
      $order_done_data->getOrderDoneTempData()
    );
    $result_data['data'] = $order_done_data->getClientData();
    $this->deleteKey($order_ref);
    return $result_data;
  }

  /**
   * Save data to the private temp store.
   *
   * @param string $key
   *   The key to save the data.
   * @param string $type
   *   The type of data being saved.
   * @param mixed $data
   *   The data to save.
   *
   * @todo Separate order private temp store and config hash private temp store.
   */
  private function setData(string $key, string $type, mixed $data): void {
    $existing_data = $this->privateTempStore->get($key) ?? [];
    if (!is_array($existing_data)) {
      $existing_data = [];
    }
    $existing_data[$type] = $data;
    $this->privateTempStore->set($key, $existing_data);
  }

  /**
   * Retrieve data from the private temp store.
   *
   * @param string $key
   *   The key to retrieve the data.
   * @param string $type
   *   The type of data to retrieve.
   *
   * @return mixed
   *   The data associated with the key and type, or NULL if not found.
   */
  private function getData(string $key, string $type): mixed {
    $data = $this->privateTempStore->get($key) ?? [];
    if (!is_array($data)) {
      return NULL;
    }
    return $data[$type] ?? NULL;
  }

  /**
   * Delete a key from the private temp store.
   *
   * @param string $key
   *   The key to delete.
   */
  public function deleteKey(string $key): void {
    $this->privateTempStore->delete($key);
  }

  /**
   * Get the user id from the order.
   *
   * @param string $order_ref
   *   The order reference.
   *
   * @return string|null
   *   The user id or NULL if not found.
   */
  public function getOrderUserId(string $order_ref): ?string {
    $data = $this->getData($order_ref . ':only-expire', 'uid');
    return $data ? (string) $data : NULL;
  }

  /**
   * Get the order done temp data from the order.
   *
   * @param string $order_ref
   *   The order reference.
   *
   * @return array|null
   *   The order temp data or NULL if not found.
   */
  public function getOrderDoneTempData(string $order_ref): ?array {
    $data = $this->getData($order_ref . ':only-expire', 'orderDoneTempData');
    return is_array($data) ? $data : NULL;
  }

  /**
   * Get a message by short name.
   *
   * @param string $short_name
   *   The short name of the message.
   *
   * @return string
   *   The message.
   */
  public function getMessage(string $short_name): string {
    $translatable = match ($short_name) {
      "RFA1" => $this->t("Start your BankID app."),
      "RFA2" => $this->t(
        "The BankID app is not installed. Please contact your bank."
      ),
      "RFA3" => $this->t("Action cancelled. Please try again."),
      "RFA4" => $this->t(
        "An identification or signing for this personal number is already started. Please try again."
      ),
      "RFA5" => $this->t("Internal error. Please try again."),
      "RFA6" => $this->t("Action cancelled."),
      "RFA8" => $this->t(
        "The BankID app is not responding. Please check that it's started and that you have internet access. If you don't have a valid BankID you can get one from your bank. Try again."
      ),
      "RFA9" => $this->t(
        "Enter your security code in the BankID app and select Identify or Sign."
      ),
      "RFA13" => $this->t("Trying to start your BankID app."),
      "RFA14-A" => $this->t(
        "Searching for BankID, it may take a little while... If a few seconds have passed and still no BankID has been found, you probably don't have a BankID which can be used for this identification/signing on this computer. If you have a BankID card, please insert it into your card reader. If you don't have a BankID you can get one from your bank. If you have a BankID on another device you can start the BankID app on that device."
      ),
      "RFA14-B" => $this->t(
        "Searching for BankID, it may take a little while... If a few seconds have passed and still no BankID has been found, you probably don't have a BankID which can be used for this identification/signing on this device. If you don't have a BankID you can get one from your bank. If you have a BankID on another device you can start the BankID app on that device."
      ),
      "RFA15-A" => $this->t(
        "Searching for BankID:s, it may take a little while... If a few seconds have passed and still no BankID has been found, you probably don't have a BankID which can be used for this identification/signing on this computer. If you have a BankID card, please insert it into your card reader. If you don't have a BankID you can get one from your bank."
      ),
      "RFA15-B" => $this->t(
        "Searching for BankID, it may take a little while... If a few seconds have passed and still no BankID has been found, you probably don't have a BankID which can be used for this identification/signing on this device. If you don't have a BankID you can get one from your bank."
      ),
      "RFA16" => $this->t(
        "The BankID you are trying to use is blocked or too old. Please use another BankID or get a new one from your bank."
      ),
      "RFA17-A" => $this->t(
        "The BankID app couldn't be found on your computer or mobile device. Please install it and get a BankID from your bank. Install the app from your app store or https://install.bankid.com."
      ),
      "RFA17-B" => $this->t(
        "Failed to scan the QR code. Start the BankID app and scan the QR code. Check that the BankID app is up to date. If you don't have the BankID app, you need to install it and get a BankID from your bank. Install the app from your app store or https://install.bankid.com."
      ),
      "RFA18" => $this->t("Start the BankID app."),
      "RFA19" => $this->t(
        "Would you like to identify yourself or sign with a BankID on this computer, or with a Mobile BankID?"
      ),
      "RFA20" => $this->t(
        "Do you want to use BankID on this device or another device?"
      ),
      "RFA21" => $this->t("Identification or signing in progress."),
      "RFA22" => $this->t("Unknown error. Please try again."),
      "RFA23" => $this->t(
        "Process your machine-readable travel document using the BankID app."
      ),
      default => NULL,
    };

    return $translatable ? $translatable->render() : '';
  }

}
