<?php

namespace Drupal\bankid\Controller;

use Drupal\bankid\BankIDManager;
use Drupal\bankid\BankIDOrderManager;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * BankID controller.
 */
class BankIDController extends ControllerBase {

  public function __construct(
    protected BankIDOrderManager $orderManager,
    protected RequestStack $requestStack,
    protected BankIDManager $bankIDManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('bankid.order_manager'),
      $container->get('request_stack'),
      $container->get('bankid.manager'),
    );
  }

  /**
   * Initiates an order.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function start(): JsonResponse {
    $configuration_hash = $this->requestStack->getCurrentRequest()->get('configurationHash');
    $data = $this->orderManager->startOrder($configuration_hash);
    return new JsonResponse($data, headers: ['X-Drupal-Ajax-Token' => 1]);
  }

  /**
   * Collects the result of a sign or auth order using orderRef as reference.
   *
   * @param string $orderRef
   *   The order reference.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function status(string $orderRef): JsonResponse {
    $data = $this->orderManager->getOrderStatus($orderRef);
    return new JsonResponse($data, headers: ['X-Drupal-Ajax-Token' => 1]);
  }

  /**
   * Cancels an ongoing order per the users request.
   *
   * @param string $orderRef
   *   The order reference.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function cancel(string $orderRef): JsonResponse {
    $data = $this->orderManager->cancelOrder($orderRef);
    return new JsonResponse($data, headers: ['X-Drupal-Ajax-Token' => 1]);
  }

  /**
   * Opens a dialog for Mobile BankID.
   *
   * @param string|null $configuration_hash
   *   The configuration hash is used to build the dialog. Defaults to NULL.
   * @param string $nojs
   *   A flag to indicate if JavaScript is disabled. Defaults to 'nojs'.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|array
   *   An Ajax response to open the modal dialog or an array with a markup
   *   message if JavaScript is disabled.
   */
  public function openDialog(?string $configuration_hash = NULL, string $nojs = 'nojs'): AjaxResponse|array {
    if ($nojs === 'nojs') {
      return [
        '#markup' => $this->t('Please enable JavaScript to use the BankID dialog.'),
      ];
    }

    $build = $this->bankIDManager->buildDialog($configuration_hash);

    $options = [
      'closeText' => $this->t('Close'),
      'dialogClass' => 'bankid-dialog',
    ];

    $ajax_response = new AjaxResponse();
    $ajax_response->addCommand(
      new OpenModalDialogCommand($this->t('Mobile BankID'), $build, $options)
    );
    return $ajax_response;
  }

}
