<?php

namespace Drupal\bankid\Form;

use Drupal\bankid\BankIDManager;
use Drupal\bankid\Enum\OrderOperation;
use Drupal\bankid\IntegrationBase;
use Drupal\bankid\IntegrationManager;
use Drupal\bankid\ValueObjects\CaseData;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\externalauth\AuthmapInterface;
use Drupal\externalauth\ExternalAuthInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * BankID Authenticate form.
 */
class BankIDDialogForm extends FormBase {

  /**
   * The configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Constructs a new BankIDDialogForm.
   *
   * @param \Drupal\bankid\IntegrationManager $integrationManager
   *   The integration manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\externalauth\ExternalAuthInterface $externalAuth
   *   The externalauth service.
   * @param \Drupal\externalauth\AuthmapInterface $authmap
   *   The authmap service.
   * @param \Drupal\bankid\BankIDManager $bankIDManager
   *   The BankID manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected IntegrationManager $integrationManager,
    ConfigFactoryInterface $config_factory,
    protected ExternalAuthInterface $externalAuth,
    protected AuthmapInterface $authmap,
    protected BankIDManager $bankIDManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->config = $config_factory->get('bankid.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.bankid.integration'),
      $container->get('config.factory'),
      $container->get('externalauth.externalauth'),
      $container->get('externalauth.authmap'),
      $container->get('bankid.manager'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bankid_authenticate_form';
  }

  /**
   * Form constructor.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param \Drupal\bankid\Enum\OrderOperation $operation
   *   The order operation.
   * @param string $plugin_id
   *   The plugin id.
   * @param array $plugin_configuration
   *   The plugin configuration.
   * @param array|null $private_data
   *   The private data, will break if cached by pagecache..
   *
   * @return array
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state, OrderOperation $operation = OrderOperation::Auth, string $plugin_id = 'default', array $plugin_configuration = [], ?array $private_data = NULL): array {
    $form['start'] = $this->bankIDManager->buildDialogAjaxButton(new CaseData($plugin_id, $plugin_configuration, $private_data, $operation));

    $order_ref_id = 'order-ref-id';
    $bankid_submit_id = 'bankid-submit';
    $form['order_ref'] = [
      '#type' => 'hidden',
      '#attributes' => [
        'id' => $order_ref_id,
      ],
      '#default_value' => NULL,
    ];

    $form['operation'] = [
      '#type' => 'value',
      '#value' => $operation,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#attributes' => [
        'id' => $bankid_submit_id,
        'class' => [
          'login-submit',
          'visually-hidden',
        ],
      ],
      '#attached' => [
        'library' => [
          'bankid/bankid.submit-on-order-done',
        ],
        'drupalSettings' => [
          'bankidSubmitOnOrderDone' => [
            $form['start']['#configuration_hash'] => [
              'submitId' => $bankid_submit_id,
              'orderRefId' => $order_ref_id,
            ],
          ],
        ],
      ],
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $order_ref = $form_state->getValue('order_ref');

    $operation = $form_state->getValue('operation');
    if ($operation !== OrderOperation::Auth) {
      $this->messenger()->addMessage($this->t('Operation %operation complete.', ['%operation' => $operation->value]));
      return;
    }

    $uid = $this->bankIDManager->getOrderUid($order_ref);
    $account = $uid ? $this->entityTypeManager->getStorage('user')->load($uid) : NULL;

    $redirect_path = $this->config->get('redirect_path');

    if ($account instanceof UserInterface) {
      // Redirect to the destination path if it is set.  Otherwise, redirect to
      // the configured redirect path if valid or else to the user profile.
      if ($this->getRequest()->request->has('destination')) {
        $form_state->setRedirectUrl(Url::fromUserInput($this->getRequest()->request->get('destination')));
      }
      else {
        if (!empty($redirect_path) && strpos($redirect_path, '/') === 0) {
          $url = Url::fromUserInput($redirect_path);

          if ($url->isRouted()) {
            $form_state->setRedirectUrl($url);
          }
        }
        else {
          $form_state->setRedirect(
            'entity.user.canonical',
            ['user' => $account->id()]
          );
        }
      }

      $authname = $this->authmap->get($account->id(), IntegrationBase::PROVIDER_NAME);
      $this->externalAuth->userLoginFinalize($account, $authname, IntegrationBase::PROVIDER_NAME);
    }
    else {
      $this->messenger()->addError($this->t('Unrecognized username or password.'));
    }
  }

}
