<?php

namespace Drupal\bankid\Form;

use Drupal\bankid\BankIDMockClient;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for mocking data in BankID responses.
 */
class BankIDMockForm extends FormBase {

  public function __construct(
    protected BankIDMockClient $mockClient,
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('bankid.mock_client')
    );
  }

  /**
   * @inheritDoc
   */
  public function getFormId(): string {
    return 'bankid_mock_form';
  }

  public function ajaxSubmit(array &$form, FormStateInterface $form_state): AjaxResponse {
    $ajax_response = new AjaxResponse();
    // Close the details element.
    $ajax_response->addCommand(new InvokeCommand('#bankid-mock-details', 'removeAttr', ['open']));
    return $ajax_response;
  }

  /**
   * @inheritDoc
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $build = [
      '#type' => 'details',
      '#title' => $this->t('Mock'),
      '#id' => 'bankid-mock-details',
      '#group' => 'tabs',
      '#open' => FALSE,
      '#weight' => 10,
    ];

    $build['outcome'] = [
      '#type' => 'select',
      '#options' => [
        'success' => $this->t('Success'),
        'expired' => $this->t('Expired'),
        'certificate_error' => $this->t('Certificate error'),
        'cancel' => $this->t('Cancel'),
      ],
      '#title' => $this->t('Outcome'),
    ];

    $build['pin'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Personal identitynumber'),
      '#description' => $this->t('PIN to mock'),
    ];
    $build['givenName'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Given name'),
      '#description' => $this->t('Given name to mock'),
    ];
    $build['surname'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Surname'),
      '#description' => $this->t('Surname to mock'),
    ];
    $build['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Mock'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
      ],
    ];
    $form['elements'] = $build;
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $outcome = $form_state->getValue('outcome');
    if (is_string($outcome) && strlen($outcome) > 0) {
      $this->mockClient->setCollectOutcome($outcome);
    }
    $pin = $form_state->getValue('pin');
    if (is_string($pin) && strlen($pin) > 0) {
      $givenName = $form_state->getValue('givenName');
      $givenName = is_string($givenName) && strlen($givenName) > 0 ? $givenName : 'John';
      $surname = $form_state->getValue('surname');
      $surname = is_string($surname) && strlen($surname) > 0 ? $surname : 'Doe';
      $this->mockClient->setMockCompletionUser($pin, $givenName, $surname);
    }

    $form_state->setRebuild();
  }

}
