<?php

declare(strict_types=1);

namespace Drupal\bankid\Hook;

use Drupal\bankid\IntegrationBase;
use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\externalauth\AuthmapInterface;
use Drupal\filter\FilterPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements module hooks.
 */
final class Hooks implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new Hooks object.
   */
  public function __construct(
    TranslationInterface $stringTranslation,
    protected ModuleHandlerInterface $moduleHandler,
    protected AuthmapInterface $authMap,
    protected FilterPluginManager $filterManager,
    protected ConfigFactoryInterface $configFactory,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('string_translation'),
      $container->get('module_handler'),
      $container->get('externalauth.authmap'),
      $container->get('plugin.manager.filter'),
      $container->get('config.factory'),
    );
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name, RouteMatchInterface $route_match): string|\Stringable|array|null {
    if ($route_name === 'help.page.bankid') {
      $text = file_get_contents(__DIR__ . '/README.md');
      if (!$this->moduleHandler->moduleExists('markdown')) {
        return '<pre>' . Html::escape($text) . '</pre>';
      }
      else {
        // Use the Markdown filter to render the README.
        $settings = $this->configFactory->get('markdown.settings')->getRawData();
        $config = ['settings' => $settings];
        $filter = $this->filterManager->createInstance('markdown', $config);
        return $filter->process($text, 'en');
      }
    }
    return NULL;
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme($existing, $type, $theme, $path): array {
    return [
      'bankid_dialog' => [
        'render element' => 'element',
      ],
    ];
  }

  /**
   * Implements hook_validation_constraint_alter().
   *
   * Replace core's ProtectedUserFieldConstraint with a decorated version that
   * skips the validation if a user was provisioned by BankID.
   */
  #[Hook('validation_constraint_alter')]
  public function validationConstraintAlter(array &$definitions): void {
    $definitions['ProtectedUserField']['class'] = 'Drupal\bankid\Plugin\Validation\Constraint\BankIDProtectedUserFieldConstraint';
  }

  /**
   * Implements hook_form_BASE_FORM_ID_alter().
   *
   * Hide passwords fields on the user edit form if the user was provisioned via
   * BankID as they don't have a Drupal password.
   *
   * @todo Add setting to disable this.
   */
  #[Hook('form_user_form_alter')]
  public function userFormAlter(array &$form, FormStateInterface $form_state, $form_id): void {
    /** @var \Drupal\user\AccountForm $form_object */
    $form_object = $form_state->getFormObject();
    /** @var \Drupal\user\UserInterface $user */
    $user = $form_object->getEntity();

    // If we are creating a new user, abort.
    if ($user->isNew()) {
      return;
    }

    $provisioned = $this->authMap->get((int) $user->id(), IntegrationBase::PROVIDER_NAME);

    if ($provisioned) {
      $form['account']['current_pass']['#access'] = FALSE;
      $form['account']['pass']['#access'] = FALSE;
    }
  }

}
