<?php

declare(strict_types=1);

namespace Drupal\bankid\Plugin\BankIdCase;

use Drupal\bankid\Attribute\BankIDCase;
use Drupal\bankid\BankIDCasePluginBase;
use Drupal\bankid\Enum\OrderOperation;
use Drupal\bankid\IntegrationInterface;
use Drupal\bankid\IntegrationManager;
use Drupal\bankid\ValueObjects\BankIDOrderDoneData;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\externalauth\AuthmapInterface;
use Drupal\externalauth\ExternalAuthInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'default' BankID case.
 */
#[BankIDCase(
  id: 'default',
  provider: 'bankid',
  title: new TranslatableMarkup('Default BankID Case'),
  description: new TranslatableMarkup('A case that allows authenticating via BankID.'),
)]
class DefaultCase extends BankIDCasePluginBase implements
  ContainerFactoryPluginInterface {

  /**
   * The 'bankid' module configuration.
   */
  private ImmutableConfig $config;

  /**
   * The operation to perform.
   */
  private OrderOperation $operation;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected AuthmapInterface $authmap,
    protected ExternalAuthInterface $externalAuth,
    ConfigFactoryInterface $config_factory,
    protected IntegrationManager $integrationManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config_factory->get('bankid.settings');
    $this->operation = $configuration['operation'];
  }

  /**
   * {@inheritDoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('externalauth.authmap'),
      $container->get('externalauth.externalauth'),
      $container->get('config.factory'),
      $container->get('plugin.manager.bankid.integration'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function orderCompleted(array $response): BankIDOrderDoneData {
    if ($this->operation !== OrderOperation::Auth) {
      // Signing does nothing.
      return new BankIDOrderDoneData([], $response['orderRef'], NULL, []);
    }

    $plugin = $this->integrationManager->createInstance(
      $this->config->get('integration')
    );

    $account = NULL;
    if ($plugin instanceof IntegrationInterface) {
      $account = $plugin->getUser($response);

      if (is_null($account) && $this->config->get('create_user')) {
        $account = $plugin->createUser($response);
      }
    }

    return new BankIDOrderDoneData(
      [],
      $response['orderRef'],
      $account?->id() ?? NULL,
      [],
    );
  }

}
