<?php

namespace Drupal\bankid\ValueObjects;

/**
 * Order done date.
 */
class BankIDOrderDoneData {

  /**
   * Data that is sent to the client.
   *
   * Data is emitted through the JS bankid event under 'event.details.data'.
   */
  private array $clientData;

  /**
   * The order reference.
   */
  private string $orderRef;

  /**
   * User ID if a user was linked to an order.
   */
  private ?string $userId;

  /**
   * Temporary data.
   *
   * Data will persist until removed automatically by the private temporary
   * store.
   */
  private ?array $orderDoneTempData;

  /**
   * Constructs a BankIDOrderDoneData object.
   *
   * @param array $clientData
   *   Data to be sent to the JS client.
   * @param string $orderRef
   *   The order reference.
   * @param string|null $userId
   *   The user ID of the user that was linked to the order or NULL.
   * @param array $orderDoneTempData
   *   Temporary data that will persist until removed automatically by the
   *   private temporary store.
   */
  public function __construct(array $clientData, string $orderRef, ?string $userId, array $orderDoneTempData = []) {
    $this->clientData = $clientData;
    $this->orderRef = $orderRef;
    $this->userId = $userId;
    $this->orderDoneTempData = $orderDoneTempData;
  }

  /**
   * Get JS client data.
   *
   * Data to be emitted through the JS bankid event under event.details.data.
   *
   * @return array
   *   The client data.
   */
  public function getClientData(): array {
    return $this->clientData;
  }

  /**
   * Get the order reference.
   *
   * @return string
   *   The order reference.
   */
  public function getOrderRef(): string {
    return $this->orderRef;
  }

  /**
   * Get the user ID.
   *
   * This is the user ID of the user that was linked to the order, if any.
   *
   * @return string|null
   *   The user ID, or NULL if no user was linked.
   */
  public function getUserId(): ?string {
    return $this->userId;
  }

  /**
   * Get temporary data that will persist.
   *
   * Data will persist until removed automatically by the private temporary
   * store.
   *
   * This data can be used to store additional information related to the order
   * that is not part of the client data or user ID.
   *
   * @return array
   *   The temporary data.
   */
  public function getOrderDoneTempData(): array {
    return $this->orderDoneTempData;
  }

}
