<?php

namespace Drupal\barbajs_ui\Form;

use Drupal\barbajs_ui\BarbaConstants;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a form that configures Barba settings.
 */
class BarbaSettings extends ConfigFormBase {

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Constructs a SiteInformationForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   The theme handler.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, ThemeHandlerInterface $theme_handler) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->themeHandler = $theme_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('theme_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'barba_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'barbajs_ui.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get current settings.
    $config = $this->config('barbajs_ui.settings');

    // Let module handle load Barba.js library.
    $form['load'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Load Barba.js library'),
      '#default_value' => (bool) $config->get('load'),
      '#description' => $this->t(
        "If enabled, this module will attempt to load the Barba.js for your site. To prevent loading it twice,
        leave this option disabled if you're including the assets manually or through another module or theme."
      ),
    ];

    // Choose local or CDN. Local is effective only if local files exist.
    $form['method'] = [
      '#type' => 'select',
      '#title' => $this->t('Attach method'),
      '#options' => [
        'local' => $this->t('Local'),
        'cdn' => $this->t('CDN'),
      ],
      '#default_value' => $config->get('method') ?: 'cdn',
      '#description' => $this->t(
        'Select how to load the library. Local requires a valid local build.'
      ),
      '#prefix' => '<div id="method-options">',
      '#suffix' => '</div>',
    ];

    // Build variant (minified or non-minified).
    $form['build'] = [
      '#type' => 'details',
      '#title' => $this->t('Build variant'),
      '#open' => TRUE,
    ];
    $form['build']['variant'] = [
      '#type' => 'radios',
      '#title' => $this->t('Choose production or development library.'),
      '#options' => [
        0 => $this->t('Use non-minified library (Development)'),
        1 => $this->t('Use minified library (Deployment)'),
      ],
      '#default_value' => (int) $config->get('build.variant'),
      '#description' => $this->t(
        'This applies to both local and CDN attachments.'
      ),
    ];

    // Usability sections.
    $form['usability'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Usability'),
      '#parents' => ['usability_tabs'],
      '#weight' => 99,
    ];

    // File selection.
    $form['file_types'] = [
      '#type' => 'details',
      '#title' => $this->t('Files'),
      '#attributes' => ['class' => ['details--settings', 'b-tooltip']],
      '#group' => 'usability_tabs',
      '#open' => TRUE,
    ];
    $form['file_types']['core'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Attach Barba.js core'),
      '#default_value' => (bool) $config->get('file_types.core'),
      '#description' => $this->t(
        'Core must be enabled for plugins to work.'
      ),
      '#label_attributes' => [
        'data-version' => BarbaConstants::CORE_LATEST_VERSION,
        'class' => ['barba-has-version'],
      ],
    ];

    $form['file_types']['plugins'] = [
      '#type' => 'container',
      '#title' => $this->t('Barba plugins'),
      '#prefix' => '<div id="file-plugins">',
      '#suffix' => '</div>',
      '#states' => [
        'invisible' => [
          ':input[name="core"]' => ['checked' => FALSE],
        ],
      ],
    ];
    $form['file_types']['plugins']['css'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use CSS'),
      '#default_value' => (bool) $config->get('file_types.css'),
      '#description' => $this->t(
        'Toggles helper CSS classes during transitions.'
      ),
      '#label_attributes' => [
        'data-version' => BarbaConstants::CSS_PLUGIN_VERSION,
        'class' => ['barba-has-version'],
      ],
    ];
    $form['file_types']['plugins']['prefetch'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use prefetch'),
      '#default_value' => (bool) $config->get('file_types.prefetch'),
      '#description' => $this->t(
        'Prefetches and caches linked pages to speed up navigation.'
      ),
      '#label_attributes' => [
        'data-version' => BarbaConstants::PREFETCH_PLUGIN_VERSION,
        'class' => ['barba-has-version'],
      ],
    ];
    $form['file_types']['plugins']['router'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use router'),
      '#default_value' => (bool) $config->get('file_types.router'),
      '#description' => $this->t(
        'Defines custom routes to control page transitions.'
      ),
      '#label_attributes' => [
        'data-version' => BarbaConstants::ROUTER_PLUGIN_VERSION,
        'class' => ['barba-has-version'],
      ],
    ];

    // Warning for disabled all files.
    $file_warning = $this->t('If all icon styles are disabled, the library cannot load. Therefore, after saving settings, the load option will be disabled and revert to the default state.');
    $form['file_types']['file_warning'] = [
      '#type' => 'container',
      '#markup' => '<div class="file-types-report">' . $file_warning . '</div>',
      '#prefix' => '<div id="file-warning">',
      '#suffix' => '</div>',
      '#states' => [
        'visible' => [
          // All of these must be unchecked (AND).
          '[data-drupal-selector="edit-core"]' => ['checked' => FALSE],
          '[data-drupal-selector="edit-css"]' => ['checked' => FALSE],
          '[data-drupal-selector="edit-prefetch"]' => ['checked' => FALSE],
          '[data-drupal-selector="edit-router"]' => ['checked' => FALSE],
        ],
      ],
      '#weight' => -9,
    ];

    // Theme visibility.
    $themes = $this->themeHandler->listInfo();
    $active_themes = [];
    foreach ($themes as $key => $theme) {
      if (!empty($theme->status)) {
        $active_themes[$key] = $theme->info['name'];
      }
    }

    $form['theme_groups'] = [
      '#type' => 'details',
      '#title' => $this->t('Themes'),
      '#attributes' => ['class' => ['details--settings', 'b-tooltip']],
      '#group' => 'usability_tabs',
    ];
    $form['theme_groups']['themes'] = [
      '#type' => 'select',
      '#title' => $this->t('Installed themes'),
      '#description' => $this->t(
        'Restrict loading to a subset of themes.'
      ),
      '#options' => $active_themes,
      '#multiple' => TRUE,
      '#default_value' => (array) $config->get('theme_groups.themes'),
    ];
    $form['theme_groups']['theme_negate'] = [
      '#type' => 'radios',
      '#title' => $this->t('Activate on specific themes'),
      '#title_display' => 'invisible',
      '#options' => [
        0 => $this->t('All themes except those selected'),
        1 => $this->t('Only the selected themes'),
      ],
      '#default_value' => (int) $config->get('theme_groups.negate'),
    ];

    // Path visibility.
    $form['request_path'] = [
      '#type' => 'details',
      '#title' => $this->t('Pages'),
      '#attributes' => ['class' => ['details--settings', 'b-tooltip']],
      '#group' => 'usability_tabs',
    ];
    $form['request_path']['pages'] = [
      '#type' => 'textarea',
      '#title' => '<span class="element-invisible">' . $this->t('Pages') . '</span>',
      '#default_value' => _barbajs_ui_array_to_string(
        (array) $config->get('request_path.pages')
      ),
      '#description' => $this->t(
        'One path per line. Use "*" as wildcard. Example: /admin/*. %front refers to the front page.',
        ['%front' => '<front>']
      ),
    ];
    $form['request_path']['page_negate'] = [
      '#type' => 'radios',
      '#title' => $this->t('Load Barba on specific pages'),
      '#title_display' => 'invisible',
      '#options' => [
        0 => $this->t('All pages except those listed'),
        1 => $this->t('Only the listed pages'),
      ],
      '#default_value' => (int) $config->get('request_path.negate'),
    ];

    // Attach admin UI library.
    $form['#attached']['library'][] = 'barbajs_ui/barba.settings';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    // Detect empty file selection to avoid attaching nothing.
    $no_files =
      empty($values['core']) &&
      empty($values['css']) &&
      empty($values['prefetch']) &&
      empty($values['router']);

    // Prepare file_types to save.
    // If none selected, reset to defaults (core on, plugins off).
    $file_types = $no_files ? [
      'core' => TRUE,
      'css' => FALSE,
      'prefetch' => FALSE,
      'router' => FALSE,
    ] : [
      'core' => !empty($values['core']),
      'css' => !empty($values['css']),
      'prefetch' => !empty($values['prefetch']),
      'router' => !empty($values['router']),
    ];

    // Save configuration.
    $this->config('barbajs_ui.settings')
      ->set('load', $no_files ? FALSE : (bool) $values['load'])
      ->set('method', $values['method'])
      ->set('build.variant', (int) $values['variant'])
      ->set('file_types.core', (bool) $file_types['core'])
      ->set('file_types.css', (bool) $file_types['css'])
      ->set('file_types.prefetch', (bool) $file_types['prefetch'])
      ->set('file_types.router', (bool) $file_types['router'])
      ->set('theme_groups.negate', (int) ($values['theme_negate'] ?? 0))
      ->set('theme_groups.themes', (array) ($values['themes'] ?? []))
      ->set('request_path.negate', (int) ($values['page_negate'] ?? 0))
      ->set('request_path.pages', (array) _barbajs_ui_string_to_array(
        (string) ($values['pages'] ?? '')
      ))
      ->save();

    // Inform the user when a reset occurred.
    if ($no_files) {
      $this->messenger()->addWarning($this->t(
        'No files were selected. Settings were reset to defaults (core enabled, plugins disabled). Global loading was disabled.'
      ));
    }

    // Flush caches so the updated config can be checked.
    drupal_flush_all_caches();

    parent::submitForm($form, $form_state);
  }

}
