<?php

namespace Drupal\basic_ads\Controller;

use Drupal;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\TypedData\Exception\MissingDataException;
use Drupal\node\Entity\Node;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\link\Plugin\Field\FieldType\LinkItem;

/**
 * Controller for tracking ad clicks.
 */
class AdTrackingController extends ControllerBase {

    /**
     * Handles ad click tracking and redirect.
     *
     * @param int $nid
     *   The basic_ad node id.
     *
     * @return TrustedRedirectResponse
     *   Redirect to the ad's target URL.
     *
     * @throws InvalidPluginDefinitionException|PluginNotFoundException|MissingDataException
     */
    public function click(int $nid): TrustedRedirectResponse
    {
        $node = \Drupal::entityTypeManager()->getStorage('node')->load($nid);
        // Verify this is an advertisement node.
        if ($node->bundle() !== 'basic_ad') {
            throw new NotFoundHttpException();
        }

        // Verify the ad has a link field.
        /* @var Node $node */
        if (!$node->hasField('field_ad_link') || $node->get('field_ad_link')->isEmpty()) {
            throw new NotFoundHttpException();
        }

        // Get the target URL.
        $link = $node->get('field_ad_link')->first();
        if (!$link instanceof LinkItem) {
            throw new NotFoundHttpException();
        }
        $url = $link->getUrl();

        if (!$url) {
            throw new NotFoundHttpException();
        }

        // Get placement from query parameter if available.
        $placement = urldecode(Drupal::request()->query->get('placement', ''));

        // Track the click.
        basic_ads_track_click($node->id(), $placement);

        // Redirect to the target URL.
        return new TrustedRedirectResponse($url->toString(), 302);
    }

    /**
     * Handles ad view tracking.
     *
     * @param int $nid
     *   The basic_ad node id.
     *
     * @return JsonResponse
     *   JSON response confirming the view was tracked.
     * @throws InvalidPluginDefinitionException
     * @throws PluginNotFoundException
     */
    public function view(int $nid): JsonResponse
    {
        /* @var Node $node */
        $node = Drupal::entityTypeManager()->getStorage('node')->load($nid);

        // Verify this is an advertisement node.
        if (!$node || $node->bundle() !== 'basic_ad') {
            throw new NotFoundHttpException();
        }

        // Get placement from query parameter if available.
        $placement = Drupal::request()->query->get('placement', '');

        // Track the view.
        basic_ads_track_impression($node->id(), $placement);

        return new JsonResponse([
            'status' => 'success',
            'nid' => $nid,
        ]);
    }

}