<?php

namespace Drupal\basic_ads\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\Role;

/**
 * Configure Basic Ads settings.
 */
class BasicAdsSettingsForm extends ConfigFormBase {

    /**
     * {@inheritdoc}
     */
    public function getFormId(): string
    {
        return 'basic_ads_settings';
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames(): array
    {
        return ['basic_ads.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state): array
    {
        $config = $this->config('basic_ads.settings');

        $form['excluded_roles'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('Exclude roles from tracking'),
            '#description' => $this->t('Select which user roles should NOT trigger ad clicks or views. When a user with any of these roles views or clicks an ad, it will not be counted.'),
            '#options' => $this->getRoleOptions(),
            '#default_value' => $config->get('excluded_roles') ?: [],
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state): void
    {
        // Filter out unchecked values.
        $excluded_roles = array_filter($form_state->getValue('excluded_roles'));

        $this->config('basic_ads.settings')
            ->set('excluded_roles', $excluded_roles)
            ->save();

        parent::submitForm($form, $form_state);
    }

    /**
     * Get all available roles as options.
     *
     * @return array
     *   Array of role names keyed by role ID.
     */
    protected function getRoleOptions(): array
    {
        $roles = Role::loadMultiple();

        return array_map(function ($role) {
            return $role->label();
        }, $roles);
    }

}