<?php

namespace Drupal\basic_ads\Plugin\Block;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a block that renders ads for a placement term.
 *
 * @Block(
 *   id = "basic_ads_advertisement_block",
 *   admin_label = @Translation("Basic Ad block (by placement)")
 * )
 */
class AdvertisementBlock extends BlockBase {

    public function defaultConfiguration(): array {
        return ['ad_placement_term' => NULL];
    }

    /**
     * @throws InvalidPluginDefinitionException
     * @throws PluginNotFoundException
     */
    public function blockForm($form, FormStateInterface $form_state): array {
        $form['ad_placement_term'] = [
            '#type' => 'entity_autocomplete',
            '#title' => $this->t('Ad placement term'),
            '#target_type' => 'taxonomy_term',
            '#selection_settings' => ['target_bundles' => ['basic_ad_placement']],
            '#default_value' => $this->getTermEntity(),
        ];
        return $form;
    }

    /**
     * @throws InvalidPluginDefinitionException
     * @throws PluginNotFoundException
     */
    protected function getTermEntity(): ?EntityInterface
    {
        if (!empty($this->configuration['ad_placement_term'])) {
            return \Drupal::entityTypeManager()->getStorage('taxonomy_term')->load($this->configuration['ad_placement_term']);
        }
        return NULL;
    }

    public function blockSubmit($form, FormStateInterface $form_state): void {
        $term = $form_state->getValue('ad_placement_term');
        $this->configuration['ad_placement_term'] = is_array($term) ? $term[0]['target_id'] : $term;
    }

    public function build(): array {
        $term_id = $this->configuration['ad_placement_term'] ?? NULL;

        return [
            '#type' => 'view',
            '#name' => 'basic_ads_advertisements',
            '#display_id' => 'block_1',
            '#arguments' => $term_id ? [$term_id] : [],
            '#cache' => [
                // Contexts make sure the block is cached per placement term.
                'contexts' => [
                    'user',                  // If ads differ per user.
                    'url.query_args',        // Optional: if arguments can come from URL.
                ],
                'tags' => $term_id ? ['taxonomy_term:' . $term_id] : [], // Invalidate if the term changes
            ],
        ];
    }
}
