<?php

namespace Drupal\basic_ads\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for displaying ad statistics.
 */
class AdStatsController extends ControllerBase {

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * Constructs an AdStatsController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory,
    TimeInterface $time,
    Connection $database,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
    $this->time = $time;
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('logger.factory'),
      $container->get('datetime.time'),
      $container->get('database')
    );
  }

  /**
   * Displays overview of all ad statistics.
   *
   * @return array
   *   Render array for the overview page.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function overview(): array {

    // Get all advertisement nodes.
    $query = $this->entityTypeManager->getStorage('node')->getQuery()
      ->condition('type', 'basic_ad')
      ->sort('created', 'DESC')
      ->accessCheck();

    $nids = $query->execute();

    if (empty($nids)) {
      return [
        '#markup' => $this->t('No basic advertisements found.'),
      ];
    }

    $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($nids);

    $rows = [];
    foreach ($nodes as $node) {
      /** @var \Drupal\node\NodeInterface $node */
      $impressions = basic_ads_get_impression_count($node->id());
      $clicks = basic_ads_get_click_count($node->id());
      $ctr = $impressions > 0 ? round(($clicks / $impressions) * 100, 2) : 0;

      $rows[] = [
        Link::fromTextAndUrl(
          $node->getTitle(),
          Url::fromRoute('basic_ads.stats_detail', ['nid' => $node->id()])
        ),
        $node->isPublished() ? $this->t('Published') : $this->t('Unpublished'),
        number_format($impressions),
        number_format($clicks),
        $ctr . '%',
        Link::fromTextAndUrl(
          $this->t('View Details'),
          Url::fromRoute('basic_ads.stats_detail', ['nid' => $node->id()])
        ),
      ];
    }

    $build = [];

    $build['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Advertisement'),
        $this->t('Status'),
        $this->t('Impressions'),
        $this->t('Clicks'),
        $this->t('CTR'),
        $this->t('Operations'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No advertisements found.'),
    ];

    return $build;
  }

  /**
   * Displays detailed statistics for a specific ad.
   *
   * @param int $nid
   *   The advertisement node.
   *
   * @return array
   *   Render array for the detail page.
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function detail(int $nid): array {
    try {
      $node = $this->entityTypeManager->getStorage('node')->load($nid);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->loggerFactory->get('basic_ads')->warning($e->getMessage());
      throw new NotFoundHttpException();
    }

    if (!$node || $node->bundle() !== 'basic_ad') {
      throw new NotFoundHttpException();
    }

    $request_time = $this->time->getRequestTime();

    // Overall stats.
    $total_impressions = basic_ads_get_impression_count($node->id());
    $total_clicks = basic_ads_get_click_count($node->id());
    $total_ctr = $total_impressions > 0 ?
      round(($total_clicks / $total_impressions) * 100, 2) : 0;

    // Last 30 days.
    $thirty_days_ago = $request_time - (30 * 24 * 60 * 60);
    $impressions_30d = basic_ads_get_impression_count(
      $node->id(),
      $thirty_days_ago
    );
    $clicks_30d = basic_ads_get_click_count($node->id(), $thirty_days_ago);
    $ctr_30d = $impressions_30d > 0 ?
      round(($clicks_30d / $impressions_30d) * 100, 2) : 0;

    // Last 7 days.
    $seven_days_ago = $request_time - (7 * 24 * 60 * 60);
    $impressions_7d = basic_ads_get_impression_count(
      $node->id(),
      $seven_days_ago
    );
    $clicks_7d = basic_ads_get_click_count($node->id(), $seven_days_ago);
    $ctr_7d = $impressions_7d > 0 ?
      round(($clicks_7d / $impressions_7d) * 100, 2) : 0;

    // Today.
    $today_start = strtotime('today midnight');
    $impressions_today = basic_ads_get_impression_count(
      $node->id(),
      $today_start
    );
    $clicks_today = basic_ads_get_click_count($node->id(), $today_start);
    $ctr_today = $impressions_today > 0 ?
      round(($clicks_today / $impressions_today) * 100, 2) : 0;

    $build = [];

    $build['title'] = [
      '#markup' => '<h2>' . $this->t(
        'Statistics for: @title',
        ['@title' => $node->getTitle()]
      ) . '</h2>',
    ];

    $build['ad_link'] = [
      '#markup' => '<p>' . Link::fromTextAndUrl(
        $this->t('View Advertisement'),
        $node->toUrl()
      )->toString() . '</p>',
    ];

    // Summary table.
    $build['summary'] = [
      '#type' => 'table',
      '#caption' => $this->t('Summary Statistics'),
      '#header' => [
        $this->t('Period'),
        $this->t('Impressions'),
        $this->t('Clicks'),
        $this->t('CTR'),
      ],
      '#rows' => [
        [
          $this->t('Today'),
          number_format($impressions_today),
          number_format($clicks_today),
          $ctr_today . '%',
        ],
        [
          $this->t('Last 7 Days'),
          number_format($impressions_7d),
          number_format($clicks_7d),
          $ctr_7d . '%',
        ],
        [
          $this->t('Last 30 Days'),
          number_format($impressions_30d),
          number_format($clicks_30d),
          $ctr_30d . '%',
        ],
        [
          [
            'data' => $this->t('All Time'),
            'style' => 'font-weight: bold;',
          ],
          [
            'data' => number_format($total_impressions),
            'style' => 'font-weight: bold;',
          ],
          [
            'data' => number_format($total_clicks),
            'style' => 'font-weight: bold;',
          ],
          [
            'data' => $total_ctr . '%',
            'style' => 'font-weight: bold;',
          ],
        ],
      ],
    ];

    // Stats by placement.
    $placement_stats = $this->getPlacementStats($node->id());

    if (!empty($placement_stats)) {
      $placement_rows = [];
      foreach ($placement_stats as $stat) {
        $ctr = $stat->impressions > 0 ?
          round(($stat->clicks / $stat->impressions) * 100, 2) : 0;
        $placement_rows[] = [
          $stat->placement ?: $this->t('Unknown'),
          number_format($stat->impressions),
          number_format($stat->clicks),
          $ctr . '%',
        ];
      }

      $build['placement_stats'] = [
        '#type' => 'table',
        '#caption' => $this->t('Statistics by Placement'),
        '#header' => [
          $this->t('Placement'),
          $this->t('Impressions'),
          $this->t('Clicks'),
          $this->t('CTR'),
        ],
        '#rows' => $placement_rows,
      ];
    }

    return $build;
  }

  /**
   * Get statistics grouped by placement.
   *
   * @param int $nid
   *   The node ID.
   *
   * @return array
   *   Array of statistics by placement.
   */
  protected function getPlacementStats(int $nid): array {
    $database = $this->database;

    // Get impression counts by placement.
    $impressions = $database->select('basic_ads_impressions', 'i')
      ->fields('i', ['placement'])
      ->condition('i.nid', $nid)
      ->groupBy('i.placement');
    $impressions->addExpression('COUNT(*)', 'impressions');
    $impression_results = $impressions->execute()->fetchAllAssoc('placement');

    // Get click counts by placement.
    $clicks = $database->select('basic_ads_clicks', 'c')
      ->fields('c', ['placement'])
      ->condition('c.nid', $nid)
      ->groupBy('c.placement');
    $clicks->addExpression('COUNT(*)', 'clicks');
    $click_results = $clicks->execute()->fetchAllAssoc('placement');

    // Combine the results.
    $stats = [];
    $all_placements = array_unique(array_merge(
      array_keys($impression_results),
      array_keys($click_results)
    ));

    foreach ($all_placements as $placement) {
      $stat = new \stdClass();
      $stat->placement = $placement;
      $stat->impressions = isset($impression_results[$placement]) ?
        $impression_results[$placement]->impressions : 0;
      $stat->clicks = isset($click_results[$placement]) ?
        $click_results[$placement]->clicks : 0;
      $stats[] = $stat;
    }

    return $stats;
  }

}
