<?php

namespace Drupal\basic_ads\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a block that renders ads for a placement term.
 *
 * @Block(
 *   id = "basic_ads_advertisement_block",
 *   admin_label = @Translation("Basic Ad block (by placement)")
 * )
 */
class AdvertisementBlock extends BlockBase implements
  ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs an AdvertisementBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return ['ad_placement_term' => NULL];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form['ad_placement_term'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Ad placement term'),
      '#target_type' => 'taxonomy_term',
      '#selection_settings' => ['target_bundles' => ['basic_ad_placement']],
      '#default_value' => $this->getTermEntity(),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function getTermEntity(): ?EntityInterface {
    if (!empty($this->configuration['ad_placement_term'])) {
      return $this->entityTypeManager
        ->getStorage('taxonomy_term')
        ->load($this->configuration['ad_placement_term']);
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $term = $form_state->getValue('ad_placement_term');
    $this->configuration['ad_placement_term'] = is_array($term) ?
      $term[0]['target_id'] : $term;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $term_id = $this->configuration['ad_placement_term'] ?? NULL;

    return [
      '#type' => 'view',
      '#name' => 'basic_ads_advertisements',
      '#display_id' => 'block_1',
      '#arguments' => $term_id ? [$term_id] : [],
      '#cache' => [
        // Contexts make sure the block is cached per placement term.
        'contexts' => [
          // If ads differ per user.
          'user',
          // Optional: if arguments can come from URL.
          'url.query_args',
        ],
        // Invalidate if the term changes.
        'tags' => $term_id ? ['taxonomy_term:' . $term_id] : [],
      ],
    ];
  }

}
