<?php

namespace Drupal\basic_ads\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\link\Plugin\Field\FieldType\LinkItem;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for tracking ad clicks.
 */
class AdTrackingController extends ControllerBase {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs an AdTrackingController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    RequestStack $request_stack,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('request_stack')
    );
  }

  /**
   * Handles ad click tracking and redirect.
   *
   * @param int $nid
   *   The basic_ad node id.
   *
   * @return \Drupal\Core\Routing\TrustedRedirectResponse
   *   Redirect to the ad's target URL.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function click(int $nid): TrustedRedirectResponse {
    $node = $this->entityTypeManager->getStorage('node')->load($nid);
    // Verify this is an advertisement node.
    if ($node->bundle() !== 'basic_ad') {
      throw new NotFoundHttpException();
    }

    // Verify the ad has a link field.
    /** @var \Drupal\node\Entity\Node $node */
    if (!$node->hasField('field_ad_link') ||
      $node->get('field_ad_link')->isEmpty()) {
      throw new NotFoundHttpException();
    }

    // Get the target URL.
    $link = $node->get('field_ad_link')->first();
    if (!$link instanceof LinkItem) {
      throw new NotFoundHttpException();
    }
    $url = $link->getUrl();

    if (!$url) {
      throw new NotFoundHttpException();
    }

    // Get placement from query parameter if available.
    $placement = urldecode(
      $this->requestStack->getCurrentRequest()->query->get('placement', '')
    );

    // Track the click.
    basic_ads_track_click($node->id(), $placement);

    // Redirect to the target URL.
    return new TrustedRedirectResponse($url->toString(), 302);
  }

  /**
   * Handles ad view tracking.
   *
   * @param int $nid
   *   The basic_ad node id.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response confirming the view was tracked.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function view(int $nid): JsonResponse {
    /** @var \Drupal\node\Entity\Node $node */
    $node = $this->entityTypeManager->getStorage('node')->load($nid);

    // Verify this is an advertisement node.
    if (!$node || $node->bundle() !== 'basic_ad') {
      throw new NotFoundHttpException();
    }

    // Get placement from query parameter if available.
    $placement = $this->requestStack
      ->getCurrentRequest()
      ->query
      ->get('placement', '');

    // Track the view.
    basic_ads_track_impression($node->id(), $placement);

    return new JsonResponse([
      'status' => 'success',
      'nid' => $nid,
    ]);
  }

}
