<?php

namespace Drupal\basic_ads\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Basic Ads settings.
 */
class BasicAdsSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a BasicAdsSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'basic_ads_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['basic_ads.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $config = $this->config('basic_ads.settings');

    $form['excluded_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Exclude roles from tracking'),
      '#description' => $this->t(
        'Select which user roles should NOT trigger ad clicks or views.'
      ) . ' ' . $this->t(
        'When a user with any of these roles views or clicks an ad,'
      ) . ' ' . $this->t('it will not be counted.'),
      '#options' => $this->getRoleOptions(),
      '#default_value' => $config->get('excluded_roles') ?: [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    // Filter out unchecked values.
    $excluded_roles = array_filter($form_state->getValue('excluded_roles'));

    $this->config('basic_ads.settings')
      ->set('excluded_roles', $excluded_roles)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get all available roles as options.
   *
   * @return array
   *   Array of role names keyed by role ID.
   */
  protected function getRoleOptions(): array {
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();

    return array_map(function ($role) {
      return $role->label();
    }, $roles);
  }

}
