<?php

namespace Drupal\basic_cart\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteObjectInterface;
use Drupal\Core\Url;
use Drupal\basic_cart\Utility;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for basic_cart pages and AJAX endpoints.
 *
 * Handles cart page, add/remove actions, checkout and order creation pages.
 */
class CartController extends ControllerBase {

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The page cache kill switch service.
   *
   * @var object
   */
  protected $pageCacheKillSwitch;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * CartController constructor.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param object $page_cache_kill_switch
   *   The page_cache_kill_switch service.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(RequestStack $request_stack, $page_cache_kill_switch, FormBuilderInterface $form_builder, RendererInterface $renderer, EntityTypeManagerInterface $entity_type_manager) {
    $this->requestStack = $request_stack;
    $this->pageCacheKillSwitch = $page_cache_kill_switch;
    $this->formBuilder = $form_builder;
    $this->renderer = $renderer;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('page_cache_kill_switch'),
      $container->get('form_builder'),
      $container->get('renderer'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Get title of cart page.
   *
   * @return string
   *   The title string from configuration.
   */
  public function getCartPageTitle() {
    $config = Utility::cartSettings();
    $message = $config->get('cart_page_title');
    // Title comes from configuration (admin-provided); do not call t() on it.
    return (string) $message;
  }

  /**
   * Cart Page.
   *
   * @return array
   *   Returns Drupal cart form render array or a markup render array.
   */
  public function cart() {
    $this->pageCacheKillSwitch->trigger();

    $cart = Utility::getCart();
    $config = Utility::cartSettings();

    $request = $this->requestStack->getCurrentRequest();
    if ($route = $request->attributes->get(RouteObjectInterface::ROUTE_OBJECT)) {
      $route->setDefault('_title', $config->get('cart_page_title'));
    }

    if (!empty($cart['cart'])) {
      return $this->formBuilder->getForm('\Drupal\basic_cart\Form\CartForm');
    }

    return [
      '#type' => 'markup',
      '#markup' => (string) $config->get('empty_cart'),
    ];
  }

  /**
   * Remove node from cart.
   *
   * @param int $nid
   *   Node id of the cart content.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect back to the referrer.
   */
  public function removeFromCart($nid) {
    $this->pageCacheKillSwitch->trigger();
    Utility::removeFromCart($nid);

    // Use the request to read HTTP_REFERER instead of global.
    $request = $this->requestStack->getCurrentRequest();
    $referer = $request->headers->get('referer');
    return new RedirectResponse(Url::fromUri($referer)->toString());
  }

  /**
   * Add node to cart (AJAX).
   *
   * @param int $nid
   *   Node id of the cart content.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Json object response with status, count and HTML.
   */
  public function addToCart($nid) {
    $this->pageCacheKillSwitch->trigger();

    $request = $this->requestStack->getCurrentRequest();
    $query = $request->query;
    $config = Utility::cartSettings();

    $param['entitytype'] = $query->get('entitytype') ? $query->get('entitytype') : 'node';
    $param['quantity'] = $query->get('quantity') ? (is_numeric($query->get('quantity')) ? (int) $query->get('quantity') : 1) : 1;

    Utility::addToCart($nid, $param);

    // If a redirect is configured, the redirect handling is done elsewhere.
    if ($config->get('add_to_cart_redirect') != '<none>' && trim($config->get('add_to_cart_redirect'))) {
      // Intentionally left blank to preserve existing redirect behavior.
    }
    else {
      // Consume and show messages; return JSON with rendered cart block and
      // counts.
      $this->messenger()->all();
      $response = new \stdClass();
      $response->status = TRUE;
      $response->count = Utility::cartCount();
      $response->text = '<p class="messages messages--status">' . (string) $config->get('added_to_cart_message') . '</p>';
      $response->id = 'ajax-addtocart-message-' . $nid;
      $response->block = Utility::render('basic_cart_cart_template');
      return new JsonResponse($response);
    }

    // Fallback empty JSON response.
    return new JsonResponse([]);
  }

  /**
   * Checkout Page.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   Returns Drupal checkout form render array or a redirect response.
   */
  public function checkout() {
    $cart = Utility::getCart();
    if (isset($cart['cart']) && !empty($cart['cart'])) {
      $type = $this->entityTypeManager->getStorage('node_type')->load('basic_cart_order');
      $node = $this->entityTypeManager->getStorage('node')->create([
        'type' => $type->id(),
      ]);

      $node_create_form = $this->entityFormBuilder()->getForm($node);

      return [
        '#type' => 'markup',
        '#markup' => $this->renderer->render($node_create_form),
      ];
    }

    $url = new Url('basic_cart.cart');
    return new RedirectResponse($url->toString());
  }

  /**
   * Order create page with form.
   *
   * @return array
   *   Returns Drupal create form of order content type as rendered markup.
   */
  public function orderCreate() {
    $type = $this->entityTypeManager->getStorage('node_type')->load('basic_cart_order');
    $node = $this->entityTypeManager->getStorage('node')->create([
      'type' => $type->id(),
    ]);

    $node_create_form = $this->entityFormBuilder()->getForm($node);

    return [
      '#type' => 'markup',
      '#markup' => $this->renderer->render($node_create_form),
    ];
  }

  /**
   * Add node to cart and then redirect to configured page.
   *
   * @param int $nid
   *   Node id of the cart content.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response to configured redirect path.
   */
  public function addToCartNoRedirect($nid) {
    $this->pageCacheKillSwitch->trigger();

    $request = $this->requestStack->getCurrentRequest();
    $query = $request->query;
    $config = Utility::cartSettings();

    $param['entitytype'] = $query->get('entitytype') ? $query->get('entitytype') : 'node';
    $param['quantity'] = $query->get('quantity') ? (is_numeric($query->get('quantity')) ? (int) $query->get('quantity') : 1) : 1;

    Utility::addToCart($nid, $param);

    return new RedirectResponse(Url::fromUserInput('/' . trim($config->get('add_to_cart_redirect'), '/'))->toString());
  }

  /**
   * Get title of thank you page.
   *
   * @return string
   *   The title string from checkout configuration.
   */
  public function getThankyouPageTitle() {
    $config = (new Utility)->checkoutSettings();
    $message = $config->get('thankyou')['title'];
    return (string) $message;
  }

  /**
   * Thankyou Page.
   *
   * @return array
   *   Returns Drupal markup containing the thank you content.
   */
  public function thankYouPage() {
    $config = (new Utility)->checkoutSettings();
    return [
      '#type' => 'markup',
      '#theme' => 'basic_cart_thank_you',
      '#basic_cart' => [
        'title' => $config->get('thankyou')['title'],
        'text' => $config->get('thankyou')['text'],
      ],
    ];
  }

}
