<?php

namespace Drupal\basic_cart\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'addtocart' formatter.
 *
 * @FieldFormatter(
 *   id = "addtocart",
 *   module = "basic_cart",
 *   label = @Translation("Add to cart"),
 *   field_types = {
 *     "addtocart"
 *   }
 * )
 */
class AddToCartFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs an AddToCartFormatter object.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The label.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct($plugin_id, $plugin_definition, $field_definition, array $settings, $label, $view_mode, array $third_party_settings, ConfigFactoryInterface $config_factory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'addtocart_wrapper_container_class' => '',
      'addtocart_button_container_class' => '',
      'addtocart_button_class' => '',
      'addtocart_message_wrapper_class' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    $entity = $items->getEntity();

    // Only render when the "add to cart" field is enabled for this entity.
    $add_to_cart_value = $entity->get('add_to_cart')->getValue()[0]['value'] ?? 0;
    if ($add_to_cart_value == 1) {
      $addtocart_wrapper_container_class = Html::escape($this->getSetting('addtocart_wrapper_container_class'));
      $addtocart_button_container_class = Html::escape($this->getSetting('addtocart_button_container_class'));
      $addtocart_button_class = Html::escape($this->getSetting('addtocart_button_class'));
      $addtocart_message_wrapper_class = Html::escape($this->getSetting('addtocart_message_wrapper_class'));

      $config = $this->configFactory->get('basic_cart.settings');

      // Build URL options in a readable, wrapped style.
      $option = [
        'query' => [
          'entitytype' => $entity->getEntityTypeId(),
          'quantity' => '',
        ],
        'absolute' => TRUE,
      ];

      // Label comes from configuration (admin-provided). Do not pass a dynamic
      // string as the first argument to t(); use it directly. Provide a
      // fallback literal translation if empty.
      $label = (string) $config->get('add_to_cart_button');
      if (trim($label) === '') {
        $label = $this->t('Add to cart');
      }

      // Create the proper Url object and add attributes for the anchor.
      if (trim((string) $config->get('add_to_cart_redirect')) !== '<none>' && trim((string) $config->get('add_to_cart_redirect')) !== '') {
        $url = Url::fromRoute('basic_cart.cartadddirect', ['nid' => $entity->id()], $option);
        $url->setOption('attributes', [
          'id' => 'forquantitydynamictext_' . $entity->id(),
          'class' => ['basic_cart-get-quantity', 'button', $addtocart_button_class],
        ]);
      }
      else {
        $url = Url::fromRoute('basic_cart.cartadd', ['nid' => $entity->id()], $option);
        $url->setOption('attributes', [
          'id' => 'forquantitydynamictext_' . $entity->id(),
          'class' => [
            'basic_cart-get-quantity',
            'button',
            'use-basic_cart-ajax',
            $addtocart_button_class,
          ],
        ]);
      }

      // Use Link object for proper escaping and attribute handling.
      $link = Link::fromTextAndUrl($label, $url)->toString();

      foreach ($items as $delta => $item) {
        $elements[$delta] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => [
              'ajax-addtocart-wrapper',
              $addtocart_message_wrapper_class,
            ],
            'id' => 'ajax-addtocart-message-' . $entity->id(),
          ],
          // Use a small markup block for the wrapper and link.
          '#prefix' => '<div class="addtocart-wrapper-container ' . $addtocart_wrapper_container_class . '"><div' . ' class="addtocart-link-class ' . $addtocart_button_container_class . '">',
          '#suffix' => '</div></div>',
          '#markup' => $link,
        ];
      }
    }

    // Attach the ajax library if anything will be rendered. Attach anyway to be
    // safe; render system will ignore if $elements is empty.
    $elements['#attached']['library'][] = 'core/drupal.ajax';

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = [];

    $element['addtocart_wrapper_container_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart wrapper class (css)'),
      '#default_value' => $this->getSetting('addtocart_wrapper_container_class'),
    ];

    $element['addtocart_button_container_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart button container class (css)'),
      '#default_value' => $this->getSetting('addtocart_button_container_class'),
    ];

    $element['addtocart_button_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart button class (css)'),
      '#default_value' => $this->getSetting('addtocart_button_class'),
    ];

    $element['addtocart_message_wrapper_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart message wrapper class (css)'),
      '#default_value' => $this->getSetting('addtocart_message_wrapper_class'),
    ];

    return $element + parent::settingsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = [
      '#markup' => $this->t('Custom css classes for add to cart'),
    ];
    return $summary;
  }

}
