<?php

namespace Drupal\basic_cart\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\basic_cart\BasicCartAssistant;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Drush commands for the basic_cart project.
 */
final class BasicCartCommands extends DrushCommands {

  use AutowireTrait;

  private const BATCH_SIZE = 50;

  /**
   * Constructs a BasicCartCommands object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    #[Autowire(service: 'basic_cart.assistant')]
    private readonly BasicCartAssistant $basicCartAssistant,
  ) {
    parent::__construct();
  }

  /**
   * Enable add to cart for all nodes of all enrolled node types.
   */
  #[CLI\Command(name: 'basic-cart:enable-add-to-cart', aliases: ['baca-en'])]
  #[CLI\Usage(name: 'basic-cart:enable-add-to-cart', description: 'Enable add to cart for all nodes of all enrolled node types.')]
  public function enableAddToCart() {
    $nids = $this->basicCartAssistant->getEnabledTypesNids();
    if (empty($nids)) {
      $this->logger()->notice('No nodes found to process.');
      return;
    }

    $node_storage = $this->entityTypeManager->getStorage('node');
    $chunks = array_chunk($nids, self::BATCH_SIZE);
    $this->io()->progressStart(count($nids));
    $processed = 0;
    foreach ($chunks as $chunk) {
      $nodes = $node_storage->loadMultiple($chunk);
      foreach ($nodes as $node) {
        $this->basicCartAssistant->enableAddToCartForNode($node);
        $processed++;
        $this->io()->progressAdvance(1);
      }
      // Free memory from the last chunk.
      unset($nodes);
      gc_collect_cycles();
    }
    $this->io()->progressFinish();

    $this->logger()->success(dt('Processed @count nodes in batches of @batch_size.', [
      '@count' => $processed,
      '@batch_size' => self::BATCH_SIZE,
    ]));
  }

}
