<?php

namespace Drupal\basic_cart;

use Drupal\Core\Session\AccountProxyInterface;

/**
 * Session-backed cart implementation.
 *
 * Stores cart items and quantities in $_SESSION['basic_cart'] and provides the
 * same public API as CartTable so the rest of the module can be
 * storage-agnostic.
 */
class CartSession implements CartInterface {

  /**
   * The current user (account proxy).
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $user;

  /**
   * The current user id.
   *
   * @var int
   */
  protected $userId;

  /**
   * Construct a CartSession object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $user
   *   The current user (account proxy) from the container.
   */
  public function __construct(AccountProxyInterface $user) {
    $this->user = $user;
    $this->userId = $user->id();
  }

  /**
   * Function for shopping cart retrieval.
   *
   * @param int|null $nid
   *   (optional) Node id to return only a single item.
   *
   * @return array
   *   An array with keys:
   *   - cart: array|object The cart items (or single item when $nid provided).
   *   - cart_quantity: array The quantities keyed by item id.
   *   Returns empty arrays when nothing is in the cart.
   */
  public function getCart($nid = NULL) {
    if (isset($nid)) {
      return [
        'cart' => $_SESSION['basic_cart']['cart'][$nid] ?? [],
        'cart_quantity' => $_SESSION['basic_cart']['cart_quantity'][$nid] ?? [],
      ];
    }

    if (isset($_SESSION['basic_cart']['cart'])) {
      return [
        'cart' => $_SESSION['basic_cart']['cart'],
        'cart_quantity' => $_SESSION['basic_cart']['cart_quantity'],
      ];
    }

    // Empty cart.
    return [
      'cart' => [],
      'cart_quantity' => [],
    ];
  }

  /**
   * Callback function for cart/remove/.
   *
   * @param int $nid
   *   We are using the node id to remove the node in the shopping cart.
   */
  public function removeFromCart($nid) {
    $nid = (int) $nid;
    if ($nid > 0) {
      unset($_SESSION['basic_cart']['cart'][$nid]);
      unset($_SESSION['basic_cart']['cart_quantity'][$nid]);
    }
  }

  /**
   * Shopping cart reset.
   */
  public function emptyCart() {
    unset($_SESSION['basic_cart']['cart']);
    unset($_SESSION['basic_cart']['cart_quantity']);
  }

  /**
   * Add to cart.
   *
   * @param int $id
   *   Node id.
   * @param array $params
   *   Quantity and entity types. Expected keys:
   *   - quantity: int
   *   - entitytype: string.
   */
  public function addToCart($id, array $params = []) {
    $config = Settings::cartSettings();
    if (!empty($params)) {
      $quantity = isset($params['quantity']) ? (int) $params['quantity'] : 0;
      $entitytype = $params['entitytype'] ?? 'node';

      if ($id > 0 && $quantity > 0) {
        // If a node is added more times, just update the quantity.
        $cart = $this->getCart();
        if ($config->get('quantity_status') && !empty($cart['cart']) && in_array($id, array_keys($cart['cart']))) {
          // Clicked multiple times on add to cart button. Increment quantity.
          $_SESSION['basic_cart']['cart_quantity'][$id] += $quantity;
        }
        else {
          $entity = \Drupal::entityTypeManager()->getStorage($entitytype)->load($id);
          $_SESSION['basic_cart']['cart'][$id] = $entity;
          $_SESSION['basic_cart']['cart_quantity'][$id] = $quantity;
        }
      }
      Settings::cartUpdatedMessage();
    }
  }

  /**
   * Login action sync.
   */
  public function loggedInActionCart() {
    return TRUE;
  }

}
