<?php

namespace Drupal\basic_cart\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'addtocartwithquantity' formatter.
 *
 * @FieldFormatter(
 *   id = "addtocartwithquantity",
 *   module = "basic_cart",
 *   label = @Translation("Add to cart with quantity"),
 *   field_types = {
 *     "addtocart"
 *   }
 * )
 */
class AddToCartWithQuantityFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs an AddToCartWithQuantityFormatter object.
   *
   * @param string $plugin_id
   *   The plugin id.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The label.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Third party settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   */
  public function __construct($plugin_id, $plugin_definition, $field_definition, array $settings, $label, $view_mode, array $third_party_settings, ConfigFactoryInterface $config_factory) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'quantity_addtocart_wrapper_container_class' => '',
      'quantity_addtocart_button_container_class' => '',
      'quantity_addtocart_button_class' => '',
      'quantity_addtocart_message_wrapper_class' => '',
      'quantity_addtocart_quantity_wrapper_container_class' => '',
      'quantity_addtocart_quantity_textfield_class' => '',
      'quantity_addtocart_quantity_label_class' => '',
      'quantity_addtocart_quantity_label_value' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $config = $this->configFactory->get('basic_cart.settings');

    $addtocart_wrapper_container_class = Html::escape($this->getSetting('quantity_addtocart_wrapper_container_class'));
    $addtocart_button_container_class = Html::escape($this->getSetting('quantity_addtocart_button_container_class'));
    $addtocart_button_class = Html::escape($this->getSetting('quantity_addtocart_button_class'));
    $addtocart_message_wrapper_class = Html::escape($this->getSetting('quantity_addtocart_message_wrapper_class'));
    $addtocart_quantity_wrapper_container_class = Html::escape($this->getSetting('quantity_addtocart_quantity_wrapper_container_class'));
    $addtocart_quantity_textfield_class = Html::escape($this->getSetting('quantity_addtocart_quantity_textfield_class'));
    $addtocart_quantity_label_class = Html::escape($this->getSetting('quantity_addtocart_quantity_label_class'));

    // Admin-provided label value: do not pass a dynamic string directly to t().
    $raw_label_value = (string) $this->getSetting('quantity_addtocart_quantity_label_value');
    $addtocart_quantity_label_value = Html::escape($raw_label_value) !== '' ? Html::escape($raw_label_value) : $this->t('Quantity');

    $entity = $items->getEntity();
    $elements = [];

    $option = [
      'query' => [
        'entitytype' => $entity->getEntityTypeId(),
        'quantity' => '',
      ],
      'absolute' => TRUE,
    ];

    $label = (string) $config->get('add_to_cart_button');
    if (trim($label) === '') {
      $label = $this->t('Add to cart');
    }

    if (trim((string) $config->get('add_to_cart_redirect')) !== '<none>' && trim((string) $config->get('add_to_cart_redirect')) !== '') {
      $url = Url::fromRoute('basic_cart.cartadddirect', ['nid' => $entity->id()], $option);
      $url->setOption('attributes', [
        'id' => 'forquantitydynamictext_' . $entity->id(),
        'class' => ['basic_cart-get-quantity', 'button', $addtocart_button_class],
      ]);
    }
    else {
      $url = Url::fromRoute('basic_cart.cartadd', ['nid' => $entity->id()], $option);
      $url->setOption('attributes', [
        'id' => 'forquantitydynamictext_' . $entity->id(),
        'class' => [
          'basic_cart-get-quantity',
          'button',
          'use-basic_cart-ajax',
          $addtocart_button_class,
        ],
      ]);
    }

    $link = Link::fromTextAndUrl($label, $url)->toString();

    $quantity_content = $config->get('quantity_status')
      ? '<div id="quantity-wrapper_' . $entity->id() . '" class="addtocart-quantity-wrapper-container ' . $addtocart_quantity_wrapper_container_class . '"></div>'
      : '';

    foreach ($items as $delta => $item) {
      $elements[$delta] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['ajax-addtocart-wrapper', $addtocart_message_wrapper_class],
          'id' => 'ajax-addtocart-message-' . $entity->id(),
        ],
        '#prefix' => '<div class="addtocart-wrapper-container ' . $addtocart_wrapper_container_class . '">' . $quantity_content . '<div class="addtocart-link-class ' . $addtocart_button_container_class . '">',
        '#suffix' => '</div></div>',
        '#markup' => $link,
      ];
    }

    $elements['#attached']['library'][] = 'core/drupal.ajax';
    $elements['#attached']['drupalSettings']['basic_cart']['textfield_class'] = $addtocart_quantity_textfield_class;
    $elements['#attached']['drupalSettings']['basic_cart']['label_class'] = $addtocart_quantity_label_class;
    $elements['#attached']['drupalSettings']['basic_cart']['label_value'] = $addtocart_quantity_label_value;

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = [];

    $element['quantity_addtocart_wrapper_container_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart wrapper class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_wrapper_container_class'),
    ];
    $element['quantity_addtocart_button_container_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart button container class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_button_container_class'),
    ];
    $element['quantity_addtocart_button_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart button class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_button_class'),
    ];
    $element['quantity_addtocart_message_wrapper_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add to cart message wrapper class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_message_wrapper_class'),
    ];
    $element['quantity_addtocart_quantity_wrapper_container_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Quantity wrapper class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_quantity_wrapper_container_class'),
    ];
    $element['quantity_addtocart_quantity_textfield_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Quantity textfield class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_quantity_textfield_class'),
    ];
    $element['quantity_addtocart_quantity_label_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Quantity label class (css)'),
      '#default_value' => $this->getSetting('quantity_addtocart_quantity_label_class'),
    ];
    $element['quantity_addtocart_quantity_label_value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Quantity label value'),
      '#default_value' => $this->getSetting('quantity_addtocart_quantity_label_value'),
    ];

    return $element + parent::settingsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = ['#markup' => $this->t('Custom css classes for add to cart')];
    return $summary;
  }

}
