<?php

namespace Drupal\basic_cart;

use Drupal\Core\Session\AccountProxyInterface;

/**
 * Provides a cart implementation that persists items to the DB table.
 *
 * This class syncs session-based cart data with the basic_cart_cart table via
 * CartStorage, and exposes cart operations used by the module.
 */
class CartTable implements CartInterface {

  /**
   * The current user (account proxy).
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $user;

  /**
   * The current user id.
   *
   * @var int
   */
  protected $userId;

  /**
   * The cart storage handler.
   *
   * @var \Drupal\basic_cart\CartStorage
   */
  private $cartStorage;

  /**
   * Carttable means now to db table.
   *
   * @param \Drupal\basic_cart\CartStorage $cartStorage
   *   Object of cart storage.
   * @param \Drupal\Core\Session\AccountProxyInterface $user
   *   The current user (account proxy) from the container.
   */
  public function __construct(CartStorage $cartStorage, AccountProxyInterface $user) {
    $this->user = $user;
    $this->userId = $user->id();
    $this->cartStorage = $cartStorage;
  }

  /**
   * Function for shopping cart retrieval.
   *
   * @param int|null $nid
   *   (optional) Node id to return only a single item.
   *
   * @return array
   *   An array with keys:
   *   - cart: array|object The cart items (or single item when $nid provided).
   *   - cart_quantity: array The quantities keyed by item id.
   *   Returns empty arrays when nothing is in the cart.
   */
  public function getCart($nid = NULL) {
    if (isset($nid)) {
      return [
        'cart' => $_SESSION['basic_cart']['cart'][$nid],
        'cart_quantity' => $_SESSION['basic_cart']['cart_quantity'][$nid],
      ];
    }

    if (isset($_SESSION['basic_cart']['cart'])) {
      return [
        'cart' => $_SESSION['basic_cart']['cart'],
        'cart_quantity' => $_SESSION['basic_cart']['cart_quantity'],
      ];
    }

    // Empty cart.
    return [
      'cart' => [],
      'cart_quantity' => [],
    ];
  }

  /**
   * Callback function for cart/remove/.
   *
   * @param int $nid
   *   We are using the node id to remove the node in the shopping cart.
   */
  public function removeFromCart($nid) {
    $nid = (int) $nid;

    if ($nid > 0) {
      $param['id'] = $nid;
      $param['uid'] = $this->userId;
      $entity = $_SESSION['basic_cart']['cart'][$nid];
      $param['entitytype'] = $entity->getEntityTypeId();
      $this->cartStorage->delete($param);
      unset($_SESSION['basic_cart']['cart'][$nid]);
      unset($_SESSION['basic_cart']['cart_quantity'][$nid]);
    }
  }

  /**
   * Shopping cart reset.
   */
  public function emptyCart() {
    $param['uid'] = $this->userId;
    $this->cartStorage->delete($param);
    unset($_SESSION['basic_cart']['cart']);
    unset($_SESSION['basic_cart']['cart_quantity']);
  }

  /**
   * Add to cart.
   *
   * @param int $id
   *   Node id.
   * @param array $params
   *   Quantity and entity types. Expected keys:
   *   - quantity: int
   *   - entitytype: string.
   */
  public function addToCart($id, array $params = []) {
    $config = Settings::cartSettings();
    if (!empty($params)) {
      $quantity = $params['quantity'] ?? 0;
      $entitytype = $params['entitytype'] ?? NULL;

      if ($id > 0 && $quantity > 0) {
        $param['uid'] = $this->userId;
        $param['id'] = $id;
        $param['entitytype'] = $entitytype;

        // If a node is added more times, just update the quantity.
        $cart = $this->getCart();
        if ($config->get('quantity_status') && !empty($cart['cart']) && in_array($id, array_keys($cart['cart']))) {
          // Clicked multiple times on add to cart button. Increment quantity.
          $_SESSION['basic_cart']['cart_quantity'][$id] += $quantity;
          $param['quantity'] = $_SESSION['basic_cart']['cart_quantity'][$id];
          $this->cartStorage->update($param);
        }
        else {
          $entity = \Drupal::entityTypeManager()->getStorage($entitytype)->load($id);
          $_SESSION['basic_cart']['cart'][$id] = $entity;
          $_SESSION['basic_cart']['cart_quantity'][$id] = $quantity;
          $param['quantity'] = $_SESSION['basic_cart']['cart_quantity'][$id];
          $this->cartStorage->insert($param);
        }
      }
      Settings::cartUpdatedMessage();
    }
  }

  /**
   * Login action sync.
   */
  public function loggedInActionCart() {
    if (isset($_SESSION['basic_cart']['cart']) && isset($_SESSION['basic_cart']['cart_quantity'])) {
      foreach ($_SESSION['basic_cart']['cart'] as $id => $value) {
        $param = [];
        $param['uid'] = $this->userId;
        $param['id'] = $id;
        $param['entitytype'] = $value->getEntityTypeId();
        $table_data = $this->cartStorage->load($param);
        if ($table_data) {
          $param['quantity'] = $_SESSION['basic_cart']['cart_quantity'][$id];
          $table_data = $this->cartStorage->update($param);
        }
        else {
          $param['quantity'] = $_SESSION['basic_cart']['cart_quantity'][$id];
          $this->cartStorage->insert($param);
        }
      }
    }

    $param = [];
    $param['uid'] = $this->userId;
    $exist_data = $this->cartStorage->load($param);
    foreach ($exist_data as $key) {
      if (!isset($_SESSION['basic_cart']['cart'][$key->id])) {
        $_SESSION['basic_cart']['cart_quantity'][$key->id] = $key->quantity;
        $_SESSION['basic_cart']['cart'][$key->id] = \Drupal::entityTypeManager()->getStorage($key->entitytype)->load($key->id);
      }
    }
  }

}
