<?php

declare(strict_types=1);

namespace Drupal\basic_cart;

use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Basic Cart utility service.
 */
final class BasicCartAssistant {

  /**
   * Constructs an Assistant object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityDisplayRepositoryInterface $entityDisplayRepository,
  ) {
  }

  /**
   * Return enabled content types machine names in an array.
   *
   * @return array
   *   An array of content type machine names.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getEnabledTypes(): array {
    // Load all field instances that match entity type and field name.
    $field_configs = $this->entityTypeManager->getStorage('field_config')->loadByProperties([
      'entity_type' => 'node',
      'field_name' => 'add_to_cart',
    ]);

    // Each field config instance is one bundle attachment.
    $bundles = [];
    foreach ($field_configs as $field_config) {
      /** @var \Drupal\field\Entity\FieldConfig $field_config */
      $bundles[] = $field_config->getTargetBundle();
    }

    return array_unique($bundles);
  }

  /**
   * Return enabled content types node IDs in an array.
   *
   * @return array
   *   An array of node IDs.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getEnabledTypesNids(): array {
    $nids = [];
    $enabled_bundles = $this->getEnabledTypes();
    if (!empty($enabled_bundles)) {
      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->condition('type', $enabled_bundles, 'IN')
        ->accessCheck(FALSE);

      $nids = $query->execute();
    }

    return $nids;
  }

  /**
   * Enable add to cart for a given node.
   *
   * @param \Drupal\core\Entity\EntityInterface $node
   *   A node that should be enabled.
   *
   * @return void
   *   De nada.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function enableAddToCartForNode($node): void {
    $node->set('add_to_cart', TRUE);
    $node->save();
  }

  /**
   * Get an array of node view modes to be used for form #options.
   *
   * @return array
   *   An array of node view mode machine names as keys and labels as values.
   */
  public function getNodeViewModesOptions(): array {
    $view_modes = $this->entityDisplayRepository->getViewModes('node');
    $options = [];
    foreach ($view_modes as $machine_name => $info) {
      $options[$machine_name] = $info['label'];
    }
    return $options;
  }

}
