<?php

namespace Drupal\basket;

use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a factory for creating instances of temporary storage.
 */
class TempStoreFactory extends SharedTempStoreFactory {

  /**
   * The basket service instance.
   *
   * @var \Basket|\Drupal\basket\Basket
   */
  protected $basket;

  /**
   * Constructs a new instance of the class.
   *
   * @param \KeyValueExpirableFactoryInterface $storage_factory
   *   The storage factory for creating expirable storage instances.
   * @param \LockBackendInterface $lock_backend
   *   The lock backend service for handling locking mechanisms.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user account proxy interface.
   * @param \Basket $basket
   *   The basket service instance.
   * @param int $expire
   *   (optional) The expiration time in seconds.
   */
  public function __construct(KeyValueExpirableFactoryInterface $storage_factory, LockBackendInterface $lock_backend, RequestStack $request_stack, AccountProxyInterface $current_user, Basket $basket, $expire = 604800) {
    parent::__construct($storage_factory, $lock_backend, $request_stack, $current_user, $expire);
    $this->basket = $basket;
  }

  /**
   * Sets the expiration value for the current instance.
   *
   * @param mixed $expire
   *   The expiration value to set.
   */
  public function setExpire($expire) {
    $this->expire = $expire;
    return $this;
  }

  /**
   * Initializes the expiration time based on the given type.
   *
   * @param string $type
   *   The type of expiration to be set. Possible values are 'delivery',
   *   'payment', 'currency', and 'finish'.
   */
  public function initExpire($type) {
    switch ($type) {
      case'delivery':
      case'payment':
        // 1 day.
        $this->expire = 86400;
        break;

      case'currency':
      case'finish':
        // 7 day.
        $this->expire = 86400 * 7;
        break;
    }
    return $this;
  }

  /**
   * Retrieves a shared temporary storage instance for the specified collection.
   *
   * @param string $collection
   *   The name of the collection for which to retrieve the shared storage.
   * @param string|null $owner
   *   (optional) The owner identifier. Defaults to NULL, in which case the
   *   current session identifier is used.
   */
  public function get($collection, $owner = NULL) {
    $owner = $this->basket->cart()->getCookieSid();

    // Store the data for this collection in the database.
    $storage = $this->storageFactory->get("basket.tempstore.$collection.$owner");
    return new SharedTempStore($storage, $this->lockBackend, $owner, $this->requestStack, $this->currentUser, $this->expire);
  }

}
