<?php

namespace Drupal\basket\Plugin\views\cache;

use Drupal\Core\Cache\Cache;
use Drupal\views\Plugin\views\cache\Tag;
use Drupal\views\Plugin\views\filter\Bundle;

/**
 * Simple caching of query results for Views displays.
 *
 * @ingroup views_cache_plugins
 *
 * @ViewsCache(
 *   id = "basket_cache",
 *   title = @Translation("Basket goods cache"),
 * )
 */
class BasketCache extends Tag {

  /**
   * Set getCurrencyTag.
   *
   * @var string
   */
  protected $getCurrencyTag = NULL;

  /**
   * Set setCurKey.
   *
   * @var bool
   */
  protected $isSetCurrencyKey = FALSE;

  /**
   * {@inheritdoc}
   */
  public function summaryTitle() {
    return 'Basket goods cache';
  }

  /**
   * Gets an array of cache tags for the current view.
   *
   * @return string[]
   *   An array of cache tags based on the current view.
   */
  public function getCacheTags() {
    $tags = parent::getCacheTags();

    // Remove the the list cache tags for the entity types listed in this view.
    $entity_information = $this->view->getQuery()->getEntityTableInfo();
    if (!empty($entity_information)) {
      // Add the list cache tags for each entity type used by this view.
      foreach ($entity_information as $metadata) {
        $remove = \Drupal::entityTypeManager()->getDefinition($metadata['entity_type'])->getListCacheTags();
        $tags = array_diff($tags, $remove);
      }
    }

    $tags[] = $this->getCurrencyCacheTag();

    if (!isset($this->view->filter) || !is_iterable($this->view->filter)) {
      return $tags;
    }

    foreach ($this->view->filter as $filter) {
      if ($filter->getPluginId() !== 'bundle') {
        continue;
      }

      $index = array_search($filter->getEntityType() . '_list', $tags);
      if ($index !== FALSE) {
        unset($tags[$index]);
      }
      $filterTags = $this->getFilterCacheTags($filter);
      $tags = Cache::mergeTags($tags, $filterTags);
    }

    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  protected function getFilterCacheTags(Bundle $filter) {
    $cacheTags = [];

    $values = $filter->value ?? [];

    if ($filter->operator !== 'in') {
      $valueOptions = $filter->getValueOptions();
      $values = array_diff_key($valueOptions, $values);
    }

    $entityType = $filter->getEntityType();

    foreach ($values as $key => $value) {
      $cacheTags[] = $entityType . '_list:' . $key;
    }
    return $cacheTags;
  }

  /**
   * Calculates and sets a cache ID used for the result cache.
   *
   * @return string
   *   The generated cache ID.
   */
  public function generateResultsKey() {
    $this->resultsKey = parent::generateResultsKey();
    if (!$this->isSetCurrencyKey) {
      $this->resultsKey = $this->getCurrencyCacheTag() . ':' . $this->resultsKey;
      $this->isSetCurrencyKey = TRUE;
    }
    return $this->resultsKey;
  }

  /**
   * {@inheritdoc}
   */
  private function getCurrencyCacheTag() {
    if (!$this->getCurrencyTag) {
      $this->getCurrencyTag = 'basket_currency:' . \Drupal::getContainer()->get('Basket')
          ->currency()
          ->getCurrent();
    }
    return $this->getCurrencyTag;
  }

}
