<?php

namespace Drupal\basket_imex\Plugin\IMEX\field;

use Drupal\basket_imex\BasketIMEXTrait;
use Drupal\basket_imex\Plugins\IMEXfield\BasketIMEXfieldInterface;
use Drupal\Core\Url;

/**
 * MediaField IMEX type.
 *
 * @BasketIMEXfield(
 *   id = "media_files",
 *   type = {"entity_reference:media"},
 *   name = "Media files",
 *   type_info = "(string)<br/>Web link to file",
 * )
 */
class MediaField implements BasketIMEXfieldInterface {

  use BasketIMEXTrait;

  /**
   * Retrieves formatted values based on the specified field from an entity.
   *
   * @param object $entity
   *   The entity object containing the field data.
   * @param string $fieldName
   *   The field name.
   */
  public function getValues($entity, $fieldName) {

    @[$fieldName, $targetType] = explode(':', $fieldName);

    $values = [];
    if (!empty($entity->{$fieldName})) {
      $mediaList = $entity->get($fieldName)->referencedEntities();
      if (!empty($mediaList)) {
        foreach ($mediaList as $mediaItem) {

          if (!empty($targetType) && $targetType != $mediaItem->bundle()) {
            continue;
          }

          $thumbnail = $mediaItem->get('thumbnail')->first();
          if (!empty($thumbnail->entity)) {
            $url = Url::fromUri(\Drupal::service('file_url_generator')->generateAbsoluteString($thumbnail->entity->getFileUri()))->toString();
            $values[$url] = $url;
          }
        }
      }
    }

    return implode(PHP_EOL, $values);
  }

  /**
   * Processes and sets values for an entity field based.
   *
   * @param object $entity
   *   The entity whose field values need to be set or updated.
   * @param string $importValue
   *   A newline-separated string of media import values.
   */
  public function setValues($entity, $importValue = '') {
    $setValues = [];
    @[$fieldName, $targetType, $mediaFile] = explode(':', $entity->basketIMEXupdateField);

    if (empty($targetType) || empty($mediaFile)) {
      return $entity->get($fieldName)->getValue();
    }

    $mediaList = $entity->get($fieldName);
    $oldMedias = [];
    if (!empty($mediaList)) {
      foreach ($mediaList as $mediaLine) {
        $mediaItem = $mediaLine->entity;
        if (!$mediaItem) {
          continue;
        }

        $setValues[$mediaItem->id()] = [
          'target_id' => $mediaItem->id(),
        ];

        if (!empty($targetType) && $targetType != $mediaItem->bundle()) {
          continue;
        }

        $uri = !empty($mediaItem->{$mediaFile}->entity) ? $mediaItem->{$mediaFile}->entity->getFileUri() : '';
        $oldMedias[$uri] = $mediaItem;
      }
    }

    $mediaStorage = \Drupal::entityTypeManager()->getStorage('media');
    $baseMediaEntity = $mediaStorage->create([
      'bundle' => $targetType,
    ]);
    $fieldConfig = $baseMediaEntity->getFieldDefinition($mediaFile);

    if (!empty($importValue)) {
      $importValues = explode(PHP_EOL, $importValue);
      foreach ($importValues as $key => $importValue) {
        $importValue = trim($importValue);

        // Is old media.
        $url = parse_url($importValue);
        if ($url['host'] == self::getCurrentHost() && !empty($url['path'])) {
          $url['path'] = str_replace('/sites/default/files/', 'public://', $url['path']);
          if (!empty($oldMedias[$url['path']])) {
            unset($oldMedias[$url['path']]);
            continue;
          }
        }

        $fieldSettings = !empty($fieldConfig) ? $fieldConfig->getSettings() : [];
        $dir = 'public://';
        if (!empty($fieldSettings['file_directory'])) {
          $dir = 'public://' . $fieldSettings['file_directory'] . '/';
        }
        $getFid = $this->importFileByUri($importValue, $baseMediaEntity, $dir);
        if (empty($getFid)) {
          continue;
        }

        $mediaOptions = [
          'bundle' => $targetType,
          $mediaFile => [
            'target_id' => $getFid,
          ],
        ];
        $media = $mediaStorage->loadByProperties($mediaOptions);
        $media = reset($media);
        if (!$media) {
          $mediaOptions['uid'] = \Drupal::currentUser()->id();
          $media = $mediaStorage->create($mediaOptions);
          $media->setName(pathinfo($importValue, PATHINFO_BASENAME));
          $media->setPublished(TRUE);
          $media->save();
        }
        $setValues[] = [
          'target_id' => $media->id(),
        ];
      }
    }

    if (!empty($oldMedias)) {
      foreach ($oldMedias as $key => $oldMedia) {
        $id = $oldMedia->id();
        if (!empty($setValues[$id])) {
          unset($setValues[$id]);
          unset($oldMedias[$key]);
        }
      }
    }

    return array_values($setValues);
  }

  /**
   * Handles operations to be performed after saving an entity.
   */
  public function postSave($entity, $importValue = '') {

  }

}
