<?php

namespace Drupal\batch_content_sync\Batch;

use Drupal\Core\Batch\BatchBuilder;

/**
 * Batch handler
 */
class ContentSyncBatch {

  /**
   * Start a batch that pushes each node immediately.
   *
   * @param int[] $nids
   *   Node IDs to sync.
   * @param string $env
   *   Environment
   * @param string $langcode
   *   Language being pushed
   */
  public static function startBatch(array $nids, string $env, string $langcode) {
    $batch = (new BatchBuilder())
      ->setTitle(t('Syncing content to remote...'))
      ->setInitMessage(t('Starting sync.'))
      ->setProgressMessage(t('Pushed @current of @total nodes.'))
      ->setErrorMessage(t('Sync failed.'))
      ->setFinishCallback([self::class, 'onFinish']);

    foreach ($nids as $nid) {
      $batch->addOperation([self::class, 'processItem'], [$nid, $env, $langcode]);
    }

    batch_set($batch->toArray());
  }

  public static function processItem(int $nid, string $env, string $langcode, array &$context) {
    /** @var \Drupal\batch_content_sync\Service\SyncService $syncService */
    $syncService = \Drupal::service('batch_content_sync.sync_service');
    $result = $syncService->pushToRemote($nid, $env, $langcode);

    if (!empty($result['error'])) {
      $context['results']['errors'][] = "Node {$nid}: {$result['error']}";
    }
    else {
      $context['results']['success'][] = $nid;
    }

    $context['message'] = t('Processed node @nid.', ['@nid' => $nid]);
  }

  /**
   * Batch finish callback: report success or errors.
   */
  public static function onFinish(bool $success, array $results, array $operations) {
    if ($success) {
      \Drupal::messenger()->addStatus(t('All content pushed successfully.'));
    }
    else {
      \Drupal::messenger()->addError(t('Some pushes failed: @list', [
        '@list' => implode('; ', $results['errors'] ?? []),
      ]));
    }
  }
}
