<?php

namespace Drupal\batch_content_sync\Controller;

use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Controller\ControllerBase;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\media\Entity\Media;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\file\FileInterface;
use Drupal\file\Entity\File;
use Drupal\Core\File\FileSystemInterface;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionStorage\SectionStorageManagerInterface;

class ReceiverController extends ControllerBase {

  protected $entityTypeManager;
  protected $sectionStorageManager;

  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->sectionStorageManager = $container->get('plugin.manager.layout_builder.section_storage');
    return $instance;
  }

  /**
   * Returns the configured access token for validation.
   */
  private function getConfiguredToken(): string {
    return $this->config('batch_content_sync.settings')->get('access_token');
  }

  /**
   * Find or create a taxonomy term by tid, vid, name, or uuid.
   */
  protected function getOrCreateTerm($tid, $vid, $name = '', $uuid = '') {
    $storage = $this->entityTypeManager->getStorage('taxonomy_term');
    if ($uuid) {
      $found = $storage->loadByProperties(['uuid' => $uuid]);
      if ($found) return reset($found);
    }
    if ($tid) {
      $term = $storage->load($tid);
      if ($term) return $term;
    }
    if ($name && $vid) {
      $found = $storage->loadByProperties(['vid' => $vid, 'name' => $name]);
      if ($found) return reset($found);
      // Create new term if not found.
      $term = Term::create(['vid' => $vid, 'name' => $name]);
      $term->save();
      return $term;
    }
    return NULL;
  }

  /**
   * API endpoint to receive and sync entity payloads.
   */
public function receive(Request $request): JsonResponse {
  try {
    // ✅ Validate token
    $provided = $request->get('token', $request->headers->get('X-Access-Token'));
    if ($provided !== $this->getConfiguredToken()) {
      return new JsonResponse(['error' => 'Unauthorized'], 401);
    }

    // ✅ Decode JSON
    $data = json_decode($request->getContent(), TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      return new JsonResponse(['error' => 'Invalid JSON: ' . json_last_error_msg()], 400);
    }
    if (empty($data['entity_type']) || empty($data['bundle']) || !isset($data['entity'])) {
      return new JsonResponse(['error' => 'Missing required parameters.'], 400);
    }

    $etype = $data['entity_type'];
    $bundle = $data['bundle'];
    $fields = $data['entity'];

    $storage = $this->entityTypeManager->getStorage($etype);
    $entity = NULL;

    // ✅ Config & behavior
    $config = \Drupal::config('batch_content_sync.settings');
    $behavior = $config->get('existing_content_behavior') ?? 'override';

    // ✅ UUID & langcode
    $uuid = $fields['uuid']['value'] ?? $fields['uuid'] ?? null;
    $langcode = $data['langcode'] ?? ($fields['langcode'] ?? 'en');

    // ✅ Ensure language exists on target site
    $language_manager = \Drupal::languageManager();
    $existing_languages = $language_manager->getLanguages();
    if (!isset($existing_languages[$langcode])) {
      // ✅ Create language if missing
      $language_storage = \Drupal::entityTypeManager()->getStorage('configurable_language');
      $language_storage->create([
        'id' => $langcode,
        'label' => strtoupper($langcode),
      ])->save();

      \Drupal::logger('batch_content_sync')->notice('Language @langcode created automatically during sync.', [
        '@langcode' => $langcode,
      ]);
    }

    if ($uuid && $behavior === 'override') {
      // ✅ Load existing entity by UUID
      $entities = $storage->loadByProperties(['uuid' => $uuid]);
      $base_entity = $entities ? reset($entities) : NULL;

      if ($base_entity) {
        // ✅ Get translation or create it
        if ($base_entity->hasTranslation($langcode)) {
          $entity = $base_entity->getTranslation($langcode);
        } else {
          $entity = $base_entity->addTranslation($langcode);
        }
        // ✅ Prevent UUID & langcode overwrite
        unset($fields['uuid'], $fields['langcode'], $fields['nid'], $fields['vid']);
      }
    }

    // ✅ If entity does not exist, create new
    if (!$entity) {
      unset($fields['nid'], $fields['vid']);
      if ($behavior === 'clone') {
        unset($fields['uuid']);
        if (!empty($fields['title']['value'])) {
          $fields['title']['value'] = '(Clone) ' . $fields['title']['value'];
        }
      }
      $entity = $storage->create([
        'type' => $bundle,
        'langcode' => $langcode,
      ]);
    }

    // ✅ Logging
    $title = $fields['title']['value'] ? $fields['title']['value'] : 'N/A';
    \Drupal::service('batch_content_sync.logger')->log(json_encode($data), 'received', $title, $langcode);

    // ✅ Assign fields
    foreach ($fields as $field => $value) {
      if (!$entity->hasField($field)) continue;
      $definition = $entity->get($field)->getFieldDefinition();
      $type = $definition->getType();

      // Paragraph handling
      if ($type === 'entity_reference_revisions' && is_array($value)) {
        $items = [];
        foreach ($value as $p_data) {
          if (empty($p_data['type'])) continue;
          $paragraph = NULL;
          if (!empty($p_data['uuid'])) {
            $existing = \Drupal::entityTypeManager()->getStorage('paragraph')->loadByProperties(['uuid' => $p_data['uuid']]);
            $paragraph = $existing ? reset($existing) : NULL;
          }
          if (!$paragraph) {
            $paragraph = Paragraph::create(['type' => $p_data['type'], 'langcode' => $langcode]);
          }
          foreach ($p_data as $p_field => $p_value) {
            if ($paragraph->hasField($p_field)) {
              $paragraph->set($p_field, $p_value);
            }
          }
          $paragraph->save();
          $items[] = [
            'target_id' => $paragraph->id(),
            'target_revision_id' => $paragraph->getRevisionId(),
          ];
        }
        $entity->set($field, $items);
      }
      // Media handling
      elseif ($type === 'entity_reference' && $definition->getSetting('target_type') === 'media' && is_array($value)) {
        $items = [];
        foreach ($value as $m_data) {
          $media = NULL;
          // Check if media already exists by UUID
          if (!empty($m_data['uuid'])) {
            $existing = \Drupal::entityTypeManager()->getStorage('media')->loadByProperties(['uuid' => $m_data['uuid']]);
            if ($existing) {
              $media = reset($existing);
            }
          }
          // If base64 media — create via helper
          if (!empty($m_data['base64']) && !empty($m_data['filename']) && !empty($m_data['bundle']) && $media == NULL) {
            $media = $this->createMediaFromBase64($m_data);
          }
          if ($media) {
            $items[] = ['target_id' => $media->id()];
          }
        }
        $entity->set($field, $items);
      }
      // Image handling
      elseif ($type === 'image' && is_array($value) && !empty($value)) {
        $entity->set($field, $this->createImageFromBase64($value));
        continue;
      }
      // File handling.
      elseif ($type === 'file' && is_array($value) && !empty($value)) {
        $entity->set($field, $this->createFileFromBase64($value));
      }
      // Taxonomy handling
      elseif ($type === 'entity_reference' && $definition->getSetting('target_type') === 'taxonomy_term' && is_array($value)) {
        $items = [];
        foreach ($value as $term_info) {
          $term = $this->getOrCreateTerm($term_info['target_id'] ?? NULL, NULL, $term_info['name'] ?? '', $term_info['uuid'] ?? '');
          if ($term) {
            $items[] = ['target_id' => $term->id()];
          }
        }
        $entity->set($field, $items);
      }
      else {
        $entity->set($field, $value);
      }
    }

    // ✅ Assign owner
    if (!empty($fields['uid']['uuid'])) {
      $user_storage = $this->entityTypeManager->getStorage('user');
      $user = $user_storage->loadByProperties(['uuid' => $fields['uid']['uuid']]);
      $user = reset($user) ?: $user_storage->load(1);
      if ($user) {
        $entity->setOwnerId($user->id());
        $entity->setRevisionUserId($user->id());
      }
    }

    // ✅ Save entity
    $entity->save();

    // ✅ Layout builder expects the entity to be saved first.
    if (isset($fields['layout_builder__sections']) && is_array($fields['layout_builder__sections'])) {
      try {
        $section_storage = $this->sectionStorageManager->createInstance('overrides', [
          'entity_type' => 'node',
          'entity' => $entity,
        ]);
        $section_storage->setContextValue('entity', $entity);
        $section_storage->removeAllSections(); // clear existing
        foreach ($fields['layout_builder__sections'] as $section_array) {
          $section = Section::fromArray($section_array);
          $section_storage->appendSection($section);
        }
        $section_storage->save();
        $entity->save();

      } catch (\Exception $e) {
        \Drupal::logger('batch_content_sync')->warning('Failed to rebuild layout builder sections for node @nid: @msg', [
          '@nid' => $entity->id(),
          '@msg' => $e->getMessage(),
        ]);
      }
    }
    return new JsonResponse(['status' => 'ok', 'id' => $entity->id()]);

  } catch (\Exception $e) {
    \Drupal::logger('batch_content_sync')->error('Receive failed: @msg', ['@msg' => $e->getMessage()]);
    return new JsonResponse(['error' => 'Server exception: ' . $e->getMessage()], 500);
  }
}
  /**
   * Create a media entity from base64 image data.
   */
  protected function createMediaFromBase64(array $item) {
    if (empty($item['base64']) || empty($item['filename']) || empty($item['mimetype']) || empty($item['bundle'])) {
      return NULL;
    }
    $directory = 'public://sync_media';
    \Drupal::service('file_system')->prepareDirectory(
      $directory,
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );
    $uri = $directory . '/' . $item['filename'];

    // Use file.repository service instead of file_save_data()
    /** @var \Drupal\file\FileRepositoryInterface $file_repository */
    $file_repository = \Drupal::service('file.repository');
    try {
      $data = base64_decode($item['base64']);
      $file = $file_repository->writeData($data, $uri, FileSystemInterface::EXISTS_RENAME);
    } catch (\Exception $e) {
      \Drupal::logger('batch_content_sync')->error('Failed to save media file: @error', ['@error' => $e->getMessage()]);
      return NULL;
    }
    if (!$file instanceof FileInterface) {
      return NULL;
    }
    $file->setPermanent();
    $file->save();

    $media_values = [
      'bundle' => $item['bundle'],
      'field_media_image' => [
        'target_id' => $file->id(),
        'alt' => $item['filename'],
      ],
      'status' => 1,
    ];

    // ✅ Set UUID only if provided
    if (!empty($item['uuid'])) {
      $media_values['uuid'] = $item['uuid'];
    }

    $media = Media::create($media_values);
    $media->save();
    return $media;
  }
  /**
   * Creates an Image file from base64 image data.
   */
  public function createImageFromBase64(array $values) {
    $images = [];
    $file_repository = \Drupal::service('file.repository');
    foreach ($values as $img) {
      if (empty($img['base64']) || empty($img['filename']) || empty($img['mimetype'])) {
        continue;
      }
      $directory = 'public://sync_images';
      \Drupal::service('file_system')->prepareDirectory(
        $directory,
        FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
      );
      $uri = $directory . '/' . $img['filename'];

      try {
        $data = base64_decode($img['base64']);
        $file = $file_repository->writeData($data, $uri, FileSystemInterface::EXISTS_RENAME);
      } catch (\Exception $e) {
        \Drupal::logger('batch_content_sync')->error('Failed to save image file: @error', ['@error' => $e->getMessage()]);
        continue;
      }
      if ($file instanceof FileInterface) {
        $file->setPermanent();
        $file->save();
        $images[] = [
          'target_id' => $file->id(),
          'alt' => $img['alt'] ?? '',
          'title' => $img['title'] ?? '',
          'width' => $img['width'] ?? '',
          'height' => $img['height'] ?? '',
        ];
      }
    }
    return $images;
  }

  /**
   * Creates file entity from base64 file data.
   */
  public function createFileFromBase64(array $value) {
    $files = [];
    $file_repository = \Drupal::service('file.repository');
    foreach ($value as $fdata) {
      if (empty($fdata['base64']) || empty($fdata['filename']) || empty($fdata['mimetype'])) {
        continue;
      }
      $directory = 'public://sync_files';
      \Drupal::service('file_system')->prepareDirectory(
        $directory,
        FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
      );
      $uri = $directory . '/' . $fdata['filename'];

      try {
        $data = base64_decode($fdata['base64']);
        $file = $file_repository->writeData($data, $uri, FileSystemInterface::EXISTS_RENAME);
      } catch (\Exception $e) {
        \Drupal::logger('batch_content_sync')->error('Failed to save file: @error', ['@error' => $e->getMessage()]);
        continue;
      }
      if ($file instanceof FileInterface) {
        $file->setPermanent();
        $file->save();
        $files[] = ['target_id' => $file->id()];
      }
    }
    return $files;
  }
}
