<?php

namespace Drupal\batch_content_sync\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\HttpFoundation\Response;

class ContentSyncLogController extends ControllerBase {

  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  public function view() {
    $query = $this->database->select('batch_content_sync_log', 'l')
      ->fields('l', ['id', 'title', 'language', 'type', 'created'])
      ->orderBy('id', 'DESC')
      ->range(0, 100);

    $rows = [];
    foreach ($query->execute() as $record) {
      $language = \Drupal::languageManager()->getLanguage($record->language);
      $language_name = $language ? $language->getName() : $record->language;
      $rows[] = [
        'id' => $record->id,
        'title' => $record->title,
        'language' => $language_name,
        'type' => $record->type,
        'created' => \Drupal::service('date.formatter')->format($record->created, 'short'),
        'json' => '<a href="/admin/content/sync-log/' . $record->id . '" target="_blank">View</a>',
      ];
    }

    $header = ['ID', 'Title', 'Language', 'Type', 'Created', 'JSON'];

    $table_rows = '';
    foreach ($rows as $row) {
      $table_rows .= '<tr>'
        . '<td>' . $row['id'] . '</td>'
        . '<td>' . $row['title'] . '</td>'
        . '<td>' . $row['language'] . '</td>'
        . '<td>' . $row['type'] . '</td>'
        . '<td>' . $row['created'] . '</td>'
        . '<td>' . $row['json'] . '</td>'
        . '</tr>';
    }

    $html = '
      <input type="text" id="sync-log-search" placeholder="Search logs..." style="margin-bottom:10px; padding:5px; width:300px;">
      <table id="sync-log-table" class="sync-log-table" style="width:100%; border-collapse: collapse;">
        <thead><tr><th>' . implode('</th><th>', $header) . '</th></tr></thead>
        <tbody>' . $table_rows . '</tbody>
      </table>
      <script>
        const input = document.getElementById("sync-log-search");
        input.addEventListener("keyup", function() {
          const filter = input.value.toLowerCase();
          const rows = document.querySelectorAll("#sync-log-table tbody tr");
          rows.forEach(row => {
            const text = row.innerText.toLowerCase();
            row.style.display = text.includes(filter) ? "" : "none";
          });
        });
      </script>
    ';

    return [
      '#type' => 'inline_template',
      '#template' => '{{ markup|raw }}',
      '#context' => [
        'markup' => $html,
      ],
    ];
  }

public function detail($id) {
  $record = $this->database->select('batch_content_sync_log', 'l')
    ->fields('l', ['json_entity'])
    ->condition('id', $id)
    ->execute()
    ->fetchField();

  if (!$record) {
    throw new NotFoundHttpException();
  }
  $decoded = json_decode($record);
  $response = new Response(json_encode($decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
  $response->headers->set('Content-Type', 'application/json');
  return $response;
}

  public function title($id) {
    return 'Sync Log #' . $id;
  }
}