<?php

namespace Drupal\batch_content_sync\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Component\Serialization\Json;

class SyncLogger {
  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  public function log(string $json, string $type, string $title, string $langcode = 'en-gl') {
    try {
      $entityData = Json::decode($json);
    } catch (\Exception $e) {
      \Drupal::logger('batch_content_sync')->error('Failed to decode JSON: ' . $e->getMessage());
      return;
    }

    if (!is_array($entityData)) {
      \Drupal::logger('batch_content_sync')->error('Invalid entity data passed to logger.');
      return;
    }

    // Sanitize: truncate base64 data
    $this->sanitizeBase64Recursive($entityData);

    $this->database->insert('batch_content_sync_log')->fields([
      'json_entity' => Json::encode($entityData, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
      'title' => $title,
      'language' => $langcode,
      'type' => $type,
      'created' => time(),
    ])->execute();
  }

  private function sanitizeBase64Recursive(&$data) {
    if (is_array($data) || is_object($data)) {
      foreach ($data as &$value) {
        if (is_array($value) || is_object($value)) {
          $this->sanitizeBase64Recursive($value);
        } elseif (is_string($value) && (preg_match('/^data:.*;base64,/', $value) || $this->isLikelyBase64($value))) {
          $value = substr($value, 0, 30) . '...';
        }
      }
    }
  }

  private function isLikelyBase64(string $value): bool {
    return strlen($value) > 100 && preg_match('/^[A-Za-z0-9\/+]{40,}={0,2}$/', $value);
  }
}
