<?php

namespace Drupal\batch_content_sync\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Psr\Log\LoggerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;
use Symfony\Component\Serializer\SerializerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

class SyncService {
  protected $entityTypeManager;
  protected $logger;
  protected $httpClient;
  protected $serializer;
  protected $configFactory;
  protected $sectionStorageManager;

  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger,
    ClientInterface $httpClient,
    SerializerInterface $serializer,
    ConfigFactoryInterface $configFactory,
    $sectionStorageManager = NULL
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->logger            = $logger;
    $this->httpClient        = $httpClient;
    $this->serializer        = $serializer;
    $this->configFactory     = $configFactory;
    $this->sectionStorageManager = $sectionStorageManager;
  }

  /**
   * Push a full node payload (including media and paragraphs) to a remote endpoint.
   *
   * @param int $nid
   *   The node ID to push.
   * @param string $env
   *   Environment key ('qa', 'stage', 'prod').
   * @param string $langcode
   *   Language code.
   * @return array
   *   Decoded response data or error info.
   */
public function pushToRemote($nid, string $env, string $langcode): array {
  // Defensive check for nid array
  if (is_array($nid)) {
    $nid = $nid['nid'] ?? ($nid['nid[]'][0] ?? reset($nid));
  }
  if (!is_scalar($nid)) {
    return ['error' => TRUE, 'message' => 'Invalid node ID format.'];
  }

  $config = $this->configFactory->get('batch_content_sync.settings');
  $token = $config->get('access_token');
  $urlKey = $env . '_url';
  $url = $config->get($urlKey);

  $node = $this->entityTypeManager->getStorage('node')->load($nid);
  if (!$node) {
    $this->logger->error('Node {nid} not found.', ['nid' => $nid]);
    return ['error' => TRUE, 'message' => "Node {$nid} not found."];
  }

  // ✅ Load requested translation
  if ($node->hasTranslation($langcode)) {
    $node = $node->getTranslation($langcode);
  } else {
    $this->logger->warning('Translation for node {nid} not found in {langcode}', [
      'nid' => $nid,
      'langcode' => $langcode,
    ]);
    return ['error' => TRUE, 'message' => "Translation not found for language {$langcode}."];
  }

  // ✅ Prepare data
  $entity_data = $this->normalizeEntityRecursive($node);

  $payload = [
    'token' => $token,
    'entity_type' => 'node',
    'bundle' => $node->bundle(),
    'langcode' => $langcode, // ✅ Always send requested language
    'entity' => $entity_data,
  ];

  // ✅ Headers
  $headers = [
    'X-Access-Token' => $token,
    'Content-Type' => 'application/json',
    'Accept' => 'application/json',
  ];

  // ✅ Log
  $logger = \Drupal::service('batch_content_sync.logger');
  $logger->log(json_encode($payload), 'sent', $node->getTitle(), $langcode);

  // ✅ Execute request
  try {
    $response = $this->httpClient->post($url, [
      'headers' => $headers,
      'json' => $payload,
      'http_errors' => TRUE,
    ]);
    return $this->decodeResponse($response);
  }
  catch (ClientException $e) {
    $res  = $e->getResponse();
    $code = $res ? $res->getStatusCode() : 'no-response';
    $body = $res ? (string) $res->getBody() : $e->getMessage();
    $this->logger->error('Push failed ({code}): {body}', ['code' => $code, 'body' => $body]);
    return [
      'error' => TRUE,
      'status_code' => $code,
      'message' => $body,
    ];
  }
}

  /**
   * Decode JSON response safely.
   */
  protected function decodeResponse(\Psr\Http\Message\ResponseInterface $response): array {
    $body = (string) $response->getBody();
    try {
      $data = $this->serializer->decode($body, 'json');
    }
    catch (\Exception $e) {
      $this->logger->error('Response decode failed: {msg}', ['msg' => $e->getMessage()]);
      return ['status' => 'ok', 'response' => $body];
    }
    return is_array($data) ? $data : ['status' => 'ok', 'response' => $body];
  }

protected function normalizeEntityRecursive($entity) {
  $result = [
    'uuid' => $entity->uuid(),
    'bundle' => method_exists($entity, 'bundle') ? $entity->bundle() : $entity->getType(),
    'langcode' => $entity->language()->getId(), // ✅ Add language info
  ];

  foreach ($entity->getFieldDefinitions() as $field_name => $definition) {
    if (!$entity->hasField($field_name) || $entity->get($field_name)->isEmpty()) {
      continue;
    }

    // Check for core field override
    $core_value = $this->normalizeCoreField($field_name, $entity);
    if ($core_value !== NULL) {
      $result[$field_name] = $core_value;
      continue;
    }

    $type = $definition->getType();

    // Handle paragraphs
    if ($type === 'entity_reference_revisions') {
      $items = [];
      foreach ($entity->get($field_name) as $item) {
        if ($item->entity) {
          $items[] = $this->normalizeEntityRecursive($item->entity);
        }
      }
      $result[$field_name] = $items;
    }

    // Handle media and entity references
    elseif ($type === 'entity_reference') {
      $items = [];
      foreach ($entity->get($field_name) as $item) {
        if ($item->entity) {
          $target = $item->entity;

          // Media entity with image
          if ($target->getEntityTypeId() === 'media' && $target->hasField('field_media_image')) {
            $media_file = $target->get('field_media_image')->entity ?? NULL;
            if ($media_file && $media_file->hasField('uri')) {
              $mimedata = $this->getMimeMeta($media_file);
              if (file_exists($mimedata['path'])) {
                $items[] = [
                  'uuid' => $target->uuid(),
                  'bundle' => $target->bundle(),
                  'langcode' => $target->language()->getId(), // ✅ Add language for media
                  'filename' => $mimedata['filename'],
                  'mimetype' => $mimedata['mimetype'],
                  'base64' => base64_encode(file_get_contents($mimedata['path'])),
                ];
              }
            }
          } else {
            $items[] = [
              'uuid' => $target->uuid(),
              'bundle' => $target->bundle(),
              'langcode' => $target->language()->getId(), // ✅ Add language for references
            ];
          }
        }
      }
      $result[$field_name] = $items;
    }
    // Handle Image fields
    elseif ($type === 'image') {
      $images = [];
      foreach ($entity->get($field_name) as $item) {
        if ($item->entity) {
          $file = $item->entity;
          $mimedata = $this->getMimeMeta($file);
          if (file_exists($mimedata['path'])) {
            $images[] = [
              'filename' => $mimedata['filename'],
              'mimetype' => $mimedata['mimetype'],
              'base64' => base64_encode(file_get_contents($mimedata['path'])),
              'alt' => $item->alt ?? '',
              'title' => $item->title ?? '',
              'width' => $item->width ?? '',
              'height' => $item->height ?? '',
            ];
          }
        }
      }
      $result[$field_name] = $images;
    }
    // Handle file fields.
    elseif ($type === 'file') {
      $files = [];
      foreach ($entity->get($field_name) as $item) {
        if ($item->entity) {
          $file = $item->entity;
          $mimedata = $this->getMimeMeta($file);
          if (file_exists($mimedata['path'])) {
            $files[] = [
              'filename' => $mimedata['filename'],
              'mimetype' => $mimedata['mimetype'],
              'base64' => base64_encode(file_get_contents($mimedata['path'])),
            ];
          }
        }
      }
      $result[$field_name] = $files;
    }
    // Handle all other fields
    else {
      $values = $entity->get($field_name)->getValue();
      $key = $field_name;
      // 🧹 Skip layout_builder__layout sections that are empty or invalid
      if ($key === 'layout_builder__layout') {
        $sections = $entity->get($field_name)->getSections();
        $values = array_filter($sections, function ($section) {
          return method_exists($section, 'getLayoutId') && !empty($section->getLayoutId());
        });
        $values = array_values($values); // Reindex
      }
      $result[$key] = (count($values) === 1) ? $values[0] : $values;
    }
  }
  // ✅ Handle Layout Builder sections using service
  if ($this->sectionStorageManager && $entity->hasField('layout_builder__layout') && !$entity->isNew()) {
    try {
      $plugin = $this->sectionStorageManager->createInstance('overrides', [
        'entity_type' => 'node',
        'bundle' => $entity->bundle(),
        'uuid' => $entity->uuid(),
        'entity' => $entity,
      ]);
      $plugin->setContextValue('entity', $entity);
      $sections = [];
      foreach ($plugin->getSections() as $section) {
        $sections[] = $section->toArray();
      }
      if (!empty($sections)) {
        $result['layout_builder__sections'] = $sections;
      }
    } catch (\Exception $e) {
      $this->logger->warning('Layout Builder export failed for node {nid}: {msg}', [
        'nid' => $entity->id(),
        'msg' => $e->getMessage(),
      ]);
    }
  }
  return $result;
}

protected function normalizeCoreField(string $field_name, $entity) {
  switch ($field_name) {
    case 'type':
      return $entity->bundle();

    case 'uid':
      $author = $entity->getOwner();
      return $author ? [
        'uuid' => $author->uuid(),
        'name' => $author->getDisplayName(),
        'mail' => $author->getEmail(),
      ] : NULL;
    case 'revision_uid':
      $rev_uid = $entity->get('revision_uid')->entity;
      return $rev_uid ? [
        'uuid' => $rev_uid->uuid(),
        'name' => $rev_uid->getDisplayName(),
        'mail' => $rev_uid->getEmail(),
      ] : NULL;

    case 'status':
    case 'promote':
    case 'sticky':
    case 'default_langcode':
    case 'revision_default':
    case 'revision_translation_affected':
    case 'content_translation_outdated':
      return (int) $entity->get($field_name)->value;

    case 'created':
    case 'changed':
    case 'revision_timestamp':
      return (int) $entity->get($field_name)->value;

    case 'langcode':
      return $entity->language()->getId();

    default:
      return NULL;
  }
}
  /**
   * Returns the correct mimetype and filename from the payload.
   */
  protected function getMimeMeta($file): array {
    $uri = $file->getFileUri();
    $path = \Drupal::service('file_system')->realpath($uri);
    if (!file_exists($path)) {
      return [];
    }

    $mimetype = mime_content_type($path);
    $original_ext = pathinfo($file->getFilename(), PATHINFO_EXTENSION);
    $extension = preg_replace('/[^a-zA-Z0-9]/', '', explode('/', $mimetype)[1] ?? $original_ext);
    $basename = pathinfo($file->getFilename(), PATHINFO_FILENAME);
    $filename = $basename . '.' . $extension;

    return [
      'filename' => $filename,
      'mimetype' => $mimetype,
      'path' => $path,
    ];
  }

}
