<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\BatchBridge\Messenger;

use Drupal\batch_messenger\BatchBridge\BatchMessengerBatchContextManager;
use Drupal\batch_messenger\BatchMessengerMessengerWrapper;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Messenger\Exception\RecoverableMessageHandlingException;

/**
 * @internal
 */
final class LegacyBatchItemHandler {

  public function __construct(
    private BatchMessengerBatchContextManager $batchContextManager,
    private BatchMessengerMessengerWrapper $messengerWrapper,
    #[Autowire(service: 'logger.channel.batch_messenger.operation')]
    private LoggerInterface $logger,
  ) {
  }

  /**
   * @see \_batch_process()
   */
  public function __invoke(LegacyBatchItem $message): void {
    $this->batchContextManager->sandbox($message->collectionItem, function (array &$sandbox, &$isFinished) use ($message): void {
      $this->wrapSandbox($message, $sandbox, $isFinished);
    });
  }

  /**
   * @phpstan-param array<mixed> $sandbox
   */
  public function wrapSandbox(LegacyBatchItem $message, array &$sandbox, bool &$isFinished): void {
    // Assume a single pass operation and set the completion level to 1 by
    // default.
    /** @var int $finished */
    $finished = 1;

    // Append faux batch context, just like $batch_context in \_batch_process().
    $batch_context = [
      'sandbox' => &$sandbox,
      'results' => [],
      'finished' => &$finished,
      'message' => '',
    ];

    $this->messengerWrapper->wrapMessengerMessages(
      static function () use ($message, &$batch_context): void {
        // The operation runs here:
        $message($batch_context);
      },
      logger: $this->logger,
      logTemplate: 'Operation @collection:@identifier with: @message',
      logArgs: [
        '@collection' => $message->collectionItem->getCollection(),
        '@identifier' => $message->collectionItem->getIdentifier(),
      ],
    );

    if ($batch_context['message'] !== '') {
      $this->batchContextManager
        ->insertMessage(
          $message->collectionItem->getCollection(),
          $message->collectionItem->getIdentifier(),
          $batch_context['message'],
        );
    }

    // Its possible the operation mutated the result into a non-array.
    if (FALSE === \is_array($batch_context['results'])) {
      throw new \LogicException('Results must be an array.');
    }

    foreach ($batch_context['results'] as $result) {
      $this->batchContextManager
        ->insertResult(
          $message->collectionItem->getCollection(),
          $message->collectionItem->getIdentifier(),
          $result,
        );
    }

    if ($finished < 1) {
      throw new RecoverableMessageHandlingException();
    }
    else {
      $isFinished = TRUE;
    }
  }

}
