<?php

declare(strict_types=1);

namespace Drupal\batch_messenger_test;

use Drupal\Component\Datetime\TimeInterface;

/**
 * Service used to simulate time.
 */
final class TimeMachine implements TimeInterface {

  private \DateTimeInterface $time;

  private int $start;

  public function __construct(
    string $time,
    int $currentMicroSeconds = 0,
    private bool $frozen = FALSE,
  ) {
    $this->time = new \DateTimeImmutable($time);
    $this->time = $this->time->modify('+' . $currentMicroSeconds . ' microseconds');
    $this->start = $this->hrtime();
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestTime() {
    return $this->time->getTimestamp();
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestMicroTime() {
    return (float) $this->time->getTimestamp();
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentTime() {
    return $this->time->getTimestamp() + $this->offsetSinceStart();
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentMicroTime() {
    return (float) ($this->time->format('U.u') + $this->offsetSinceStart());
  }

  private function hrtime(): int {
    return (int) (\hrtime(TRUE) / 1e9);
  }

  private function offsetSinceStart(): int {
    return $this->frozen ? 0 : ($this->hrtime() - $this->start);
  }

}
