<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\BatchBridge;

use Drupal\batch_messenger\Messenger\Stamp\CollectionItem;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * @internal
 */
final class BatchMessengerBatchContextManager {

  public function __construct(
    private Connection $database,
    private TimeInterface $time,
    #[Autowire(service: 'lock')]
    private LockBackendInterface $lock,
    #[Autowire(service: 'keyvalue')]
    private KeyValueFactoryInterface $keyValueFactory,
  ) {
  }

  /**
   * @phpstan-param non-empty-string $collection
   */
  public function addBatch(string $collection, ?string $finishCallback): void {
    if (\strlen($collection) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $this->database
      ->insert('batch_messenger__batch')
      ->fields([
        'collection' => $collection,
        'finishCallback' => $finishCallback,
        'firstSeen' => $this->time->getCurrentMicroTime(),
      ])
      ->execute();
  }

  /**
   * @phpstan-param non-empty-string $collection
   */
  public function finish(string $collection): void {
    if (\strlen($collection) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $this->database
      ->update('batch_messenger__batch')
      ->fields([
        'finishedOn' => $this->time->getCurrentMicroTime(),
      ])
      ->condition('collection', $collection)
      ->execute();
  }

  /**
   * @phpstan-param non-empty-string $collection
   */
  public function getFinishCallback(string $collection): ?callable {
    if (\strlen($collection) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $query = $this->database
      ->select('batch_messenger__batch', 'b')
      ->condition('collection', $collection)
      ->fields('b', ['finishCallback'])
      ->range(0, 1);

    /** @var callable-string|false|null $value */
    $value = $query->execute()?->fetchField();
    return \is_string($value) ? $value(...) : NULL;
  }

  /**
   * @phpstan-param non-empty-string $collection
   * @phpstan-param non-empty-string $identifier
   */
  public function insertMessage(string $collection, string $identifier, mixed $message): void {
    if (\strlen($collection) > 64 || \strlen($identifier) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $this->database
      ->insert('batch_messenger__batch_messages')
      ->fields([
        'collection' => $collection,
        'identifier' => $identifier,
        'created' => $this->time->getCurrentMicroTime(),
        'message' => \serialize($message),
      ])
      ->execute();
  }

  /**
   * @phpstan-return mixed
   *   Message, or null if no message found.
   */
  public function getLatestMessage(string $collection): mixed {
    if (\strlen($collection) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $query = $this->database
      ->select('batch_messenger__batch_messages', 'm')
      ->condition('collection', $collection)
      ->fields('m', ['message'])
      ->orderBy('created', 'DESC')
      ->range(0, 1);

    /** @var string|false $value */
    $value = $query->execute()?->fetchField() ?? throw new \Exception('Bad query');
    return $value !== FALSE ? \unserialize($value) : NULL;
  }

  /**
   * @phpstan-param non-empty-string $collection
   * @phpstan-param non-empty-string $identifier
   */
  public function insertResult(string $collection, string $identifier, mixed $result): void {
    if (\strlen($collection) > 64 || \strlen($identifier) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $this->database
      ->insert('batch_messenger__batch_results')
      ->fields([
        'collection' => $collection,
        'identifier' => $identifier,
        'result' => \serialize($result),
      ])
      ->execute();
  }

  /**
   * @phpstan-return \Generator<mixed>
   */
  public function getResults(string $collection): \Generator {
    if (\strlen($collection) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $query = $this->database
      ->select('batch_messenger__batch_results', 'r')
      ->condition('collection', $collection)
      ->fields('r', ['result'])
      ->execute() ?? throw new \Exception('Bad query');

    while (($value = $query->fetchField()) !== FALSE) {
      yield $value;
    }
  }

  /**
   * @phpstan-param (callable(array<mixed>, bool&): void) $c
   */
  public function sandbox(CollectionItem $collectionItem, callable $c): void {
    $lockName = \sprintf('batch_messenger--sandbox--%s--%s', $collectionItem->getCollection(), $collectionItem->getIdentifier());
    if ($this->lock->acquire(name: $lockName, timeout: 3600)) {
      $sandboxKey = \sprintf('%s--%s', $collectionItem->getCollection(), $collectionItem->getIdentifier());
      /** @var array<mixed> $sandbox */
      $sandbox = $this->keyValueFactory
        ->get('batch_messenger-sandbox')
        // The default sandbox value is empty array:
        ->get($sandboxKey, []);
      try {
        $oldSandbox = $sandbox;
        $isFinished = FALSE;
        $c($sandbox, $isFinished);
      }
      finally {
        if ($isFinished) {
          $this->keyValueFactory->get('batch_messenger-sandbox')->delete($sandboxKey);
        }
        elseif ($oldSandbox !== $sandbox) {
          $this->keyValueFactory->get('batch_messenger-sandbox')->set($sandboxKey, $sandbox);
        }

        $this->lock->release(name: $lockName);
      }
    }
  }

}
