<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\BatchBridge;

use Drupal\batch_messenger\BatchBridge\Messenger\LegacyBatchItem;
use Drupal\batch_messenger\BatchMessengerTracker;
use Drupal\batch_messenger\Messenger\Stamp\CollectionItem;
use Drupal\Core\Database\Connection;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\MessageBusInterface;

/**
 * @internal
 */
final class BatchMessengerQueue {

  /**
   * @phpstan-param non-empty-string $name
   */
  // @phpstan-ignore-next-line
  public function __construct(
    private string $name,
    Connection $connection,
  ) {
  }

  public function createQueue(): void {
  }

  public function deleteQueue(): void {
  }

  /**
   * We use the same single operation creation logic as Batch API in createItem() but instead convert the operation to a
   * LegacyBatchItem message with a common CollectionItem stamp.
   *
   * @phpstan-param array{mixed, array<mixed>} $data
   */
  public function createItem($data): void {
    // Ensure the argument tacked on is as expected.
    $itemIdentifier = \array_pop($data[1]);
    if (!$itemIdentifier instanceof BatchMessengerItemIdentifier) {
      throw new \LogicException('Missing expected ' . BatchMessengerItemIdentifier::class);
    }

    if (\is_array($data[0])) {
      /** @var callable-string $c */
      $c = \implode('::', $data[0]);
    }
    elseif (\is_string($data)) {
      $c = $data;
    }
    else {
      throw new \LogicException('Expected an array or string of callable');
    }

    $collectionItem = CollectionItem::create($this->name, $itemIdentifier->identifier);

    static::bus()->dispatch(
      new Envelope(
        message: LegacyBatchItem::create($c, $data[1], $collectionItem),
        stamps: [$collectionItem],
      ),
    );
  }

  /**
   * @phpstan-return array<mixed>
   */
  public function getAllItems() {
    return [];
  }

  /**
   * This method returns a fake object with an operation that queries the overall processing status of Messenger,
   * and updates the set state. When _batch_process once again uses the $current_set variable-reference, it'll have the
   * updated overall state of the set.
   *
   * @see \_batch_process()
   * @phpstan-return object{data: mixed}
   */
  public function claimItem(int $lease_time = 0) {
    // Create an object consumed by _batch_process().
    return BatchMessengerClaimedItem::create(
      [BatchMessengerQueue::class, 'operationCallback'],
      [$this->name],
    );
  }

  public function deleteItem(BatchMessengerClaimedItem $item): void {
  }

  /**
   * @phpstan-param array{sandbox: array<mixed>, results: array<mixed>, finished: int, message: string} $sandbox
   */
  public static function operationCallback(string $collectionName, array &$sandbox): void {
    [$pendingCount] = static::batchMessengerTracker()->countItems($collectionName);

    // Messenger permits parallelism & out-of-order operations. The best we can
    // do is show the latest completion message. Depending on implementation,
    // this could mean messages look a bit confusing, especially if the message
    // shows anything other than the unit of work, such as trying to show
    // overall batch progress.
    $sandbox['message'] = static::batchContextManager()->getLatestMessage($collectionName);

    $sandbox['finished'] = ($pendingCount === 0) ? 1 : 0;

    // Set progress. Normally batch items don't have access to the set, so we have to hack around things.
    $current_set = &\_batch_current_set();
    // If they're already the same then don't bother setting.
    if (((int) $current_set['count']) !== $pendingCount) {
      // Always give minimum 1 so the count decrement in \_batch_process() has something to work with. This logic cannot go negative otherwise things bug out.
      $current_set['count'] = \max(1, $pendingCount);
      \_batch_queue($current_set);
    }

    static::sleep();
  }

  public static function bus(): MessageBusInterface {
    /** @var \Symfony\Component\Messenger\MessageBusInterface */
    return \Drupal::service('batch_messenger.bus.default');
  }

  public static function batchMessengerTracker(): BatchMessengerTracker {
    return \Drupal::service(BatchMessengerTracker::class);
  }

  public static function batchContextManager(): BatchMessengerBatchContextManager {
    /** @var \Drupal\batch_messenger\BatchBridge\BatchMessengerBatchContextManager */
    return \Drupal::service(BatchMessengerBatchContextManager::class);
  }

  public static function sleep(): void {
    /** @var positive-int $duration */
    // @phpstan-ignore-next-line
    $duration = (int) \Drupal::getContainer()->getParameter('batch_messenger.operation_sleep');
    \usleep($duration);
  }

}
