<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\BatchBridge\Messenger\Middleware;

use Drupal\batch_messenger\BatchBridge\BatchMessengerBatchContextManager;
use Drupal\batch_messenger\BatchBridge\Messenger\FinishBatch;
use Drupal\batch_messenger\BatchMessengerTracker;
use Drupal\batch_messenger\Messenger\Stamp\CollectionItem;
use Drupal\Core\Lock\LockBackendInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Messenger\Middleware\MiddlewareInterface;
use Symfony\Component\Messenger\Middleware\StackInterface;
use Symfony\Component\Messenger\Stamp\HandledStamp;

/**
 * Detects completion of a batch and dispatches a message to call the finish callback.
 */
final class BatchBridgePostHandleMiddleware implements MiddlewareInterface {

  public function __construct(
    private BatchMessengerTracker $batchMessengerTracker,
    private BatchMessengerBatchContextManager $batchContextManager,
    #[Autowire(service: 'lock')]
    private LockBackendInterface $lock,
    #[Autowire(service: 'batch_messenger.bus.default')]
    private MessageBusInterface $bus,
  ) {
  }

  public function handle(Envelope $envelope, StackInterface $stack): Envelope {
    if (NULL !== ($stamp = $envelope->last(CollectionItem::class))) {
      // For safety only:
      if ($envelope->last(HandledStamp::class) !== NULL) {
        // Checks if all items were handled, then kicks off a final message.
        // Ideally, this would be some kind of monitor using the Messenger
        // scheduler pieces, and maybe we can if the site opts into running one.
        [$pendingCount] = $this->batchMessengerTracker->countItems($stamp->getCollection());
        if ($pendingCount === 0) {
          // Acquire a lock for 'finishing', in case another Worker is doing the same.
          $lockName = \sprintf('batch_messenger--completed--%s', $stamp->getCollection());
          if ($this->lock->acquire($lockName)) {
            $this->batchContextManager->finish($stamp->getCollection());
            $this->bus->dispatch(FinishBatch::create($stamp->getCollection()));
            $this->lock->release($lockName);
          }
        }
      }
    }

    return $stack->next()->handle($envelope, $stack);
  }

}
