<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\Collection;

/**
 * @internal
 */
final class Collection {

  /**
   * @phpstan-param non-empty-string $collectionName
   * @phpstan-param \DateTimeInterface $created
   * @phpstan-param int<0, max> $pendingCount
   * @phpstan-param int<0, max> $processedCount
   */
  private function __construct(
    public string $collectionName,
    public \DateTimeInterface $created,
    public int $pendingCount,
    public int $processedCount,
  ) {
  }

  /**
   * @internal
   *
   * @phpstan-param non-empty-string $collectionName
   * @phpstan-param \DateTimeInterface $created
   * @phpstan-param int<0, max> $pendingCount
   * @phpstan-param int<0, max> $processedCount
   */
  public static function create(
    string $collectionName,
    \DateTimeInterface $created,
    int $pendingCount,
    int $processedCount,
  ): static {
    if (\strlen($collectionName) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    return new static(
      $collectionName,
      $created,
      $pendingCount,
      $processedCount,
    );
  }

  public function isComplete(): bool {
    return $this->pendingCount === 0;
  }

  /**
   * Returns progress between 0 and 1.
   *
   * E.g: 25% is 0.25.
   */
  public function progress(): float {
    return 1 - ($this->pendingCount / ($this->processedCount + $this->pendingCount));
  }

}
