<?php

declare(strict_types=1);

namespace Drupal\batch_messenger;

use Drupal\Core\Messenger\MessengerInterface;
use Psr\Log\LoggerInterface;

/**
 * Collects messages sent to Drupal Messenger during the lifecycle of a callback.
 *
 * @internal
 *   For internal use only.
 */
final class BatchMessengerMessengerWrapper {

  public function __construct(
    private MessengerInterface $messenger,
  ) {
  }

  /**
   * @phpstan-param array<string, mixed> $logArgs
   */
  public function wrapMessengerMessages(
    callable $c,
    LoggerInterface $logger,
    string $logTemplate = '@message',
    array $logArgs = [],
  ): void {
    if (\array_key_exists('@message', $logArgs)) {
      throw new \LogicException('Log args should not contain @message');
    }

    // Messages are deleted upfront so messages don't leak into this.
    $this->messenger->deleteAll();
    try {
      $c();
    }
    finally {
      $log = static function (string $type) use ($logger) {
        /** @var \Drupal\Core\Messenger\MessengerInterface::TYPE_* $type */
        return match ($type) {
          MessengerInterface::TYPE_ERROR => $logger->error(...),
          MessengerInterface::TYPE_STATUS => $logger->info(...),
          MessengerInterface::TYPE_WARNING => $logger->warning(...),
        };
      };

      foreach ($this->messenger->deleteAll() as $type => $messages) {
        $logger = $log($type);
        // Cores type-hint is incorrect for deleteAll().
        // @phpstan-ignore-next-line
        foreach ($messages as $m) {
          $logger($logTemplate, ['@message' => (string) $m] + $logArgs);
        }
      }
    }
  }

}
