**Batch Messenger**

https://www.drupal.org/project/batch_messenger

[[_TOC_]]

Batch Messenger allows you to group Symfony Messenger messages together into a collection, expose a UI of progress of each collection, and intercept Drupal Batch API calls (batch_set) and replace with Messenger internals.

When the legacy Batch API is used (`\batch_set`), the UI is retained, and we provide status updates to the user despite operations not actually running in the web requests.

The module is a drop-in replacement for Batch API. If you encounter issues, uninstalling the module will return existing behavior. Though, any in-progress batches will be deleted.

# API

## Collection tracking

Dispatch messages as usual, but attach a `\Drupal\batch_messenger\Messenger\Stamp\CollectionItem` stamp with a common collection ID and unique message ID. Both IDs may be randomly generated strings between 1-128 characters in length. Errors will occur if duplicate ID pairs are dispatched.

```php
use \Symfony\Component\Messenger\Envelope;
use \Drupal\batch_messenger\Messenger\Stamp\CollectionItem;

/** @var \Symfony\Component\Messenger\MessageBusInterface $bus */
$bus = \Drupal::service(MessageBusInterface::class);

// Collections:
$collectionName = 'my-set-' . \mt_rand(111,999);
foreach (range(1, 60) as $i) {
  $bus->dispatch(new Envelope(
    message: new MyMessage(),
    stamps: [
      CollectionItem::create($collection, (string) $i)
    ],
  ));
}
```

# Batch UI

A user interface showing all message collections can be found at `Administration > Reports > Message sets` (/admin/reports/batch-messenger). A dedicated permission is provided to delegate access.

A button on the _Message sets_ page is available to clean up fully executed sets.

# Batch API bridge

Batch Messenger provides an option to intercept legacy calls to Drupal's Batch API (via `\batch_set`), and convert them to Messenger messages, where each _operation_ of a batch is converted to a Messenger message.

This functionality is enabled by default. **Make sure messenger configuration is set to use an asynchronous transport**; otherwise each operation will be executed in the same request it is set on, rather than via a Messenger worker (`consume` command).

The existing Batch API UI is retained, where the user is informed of the batch progress. However, the operations are not executed within the context of a web request. The operations are executed off-thread by Messenger workers. This also means the UI does not need to be open for the batch to execute, and the user also has the option of returning to the Batch UI screen to see the progress of the batch.

## Behavioral differences

Batch Messenger does its best to simulate Batch API and the environment it runs in, but keep in mind differences:

- Since running in a messenger worker, current user and request values will not be set, or they will be simulated.
- Operation **Sandbox** (`$context['sandbox']`) is simulated:
  - When an operation does not _finish_ in a single run, a sandbox is stored and shared to the next run of the operation.
  - An operation with a sandbox can never be distributed to multiple workers. So it will be single threaded. Consider re-working operations to never make use of `$context['finished']`.
  - `$context['operations']` provided to an operation is always empty, instead of containing the real operations. This is because it does not make sense in the context of Messenger where workers can work in parallel.
  - Sandbox is only shared within a single operation, never between other operation executions.
  - The completion status or messages of an individual operation will not show on Batch API UI, until an individual operation is _finished_.
- Operation **Results**
  - Results are stored and shared to the finished callback. This works slightly different to core, in that results are not shared between operations, only ever made available to the finished callback. Operations always receive an empty set of results in `$context['results']`.
  - If there is no finished callback, results are discarded.
- **Finished** callback
  - Callback is ran in the worker after all operations complete successfully.
  - Finished callback never runs if all operations do not complete.
  - The `$success` argument for finished callback is always `TRUE`.
- Messages sent to `\Drupal::messenger()` are intercepted and sent to logs.
- Messenger does not guarantee messages run in sequence, in fact you must not ever rely on this, and you must also account for messages running in parallel. Utilize locking when necessary, and ensure each message does not write to resources that other messages write to.
- If an operation, which was converted to a message, fails. It will be sent to the 'failed' queue as configured by Messenger. A single failed message will block completion of a Message set/batch.
- When viewing the Batch API UI, messages shown to the user may appear out of order. This is because messages may execute out of order. The latest completed operation (message) is shown to the user.

# License

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
