<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\BatchBridge;

use Drupal\Component\Uuid\UuidInterface;

final class BatchBridgeHooks {

  public function __construct(
    private BatchMessengerBatchContextManager $batchContextManager,
    private UuidInterface $uuid,
  ) {
  }

  /**
   * Implements hook_batch_alter().
   *
   * This hook is a part of the BatchBridge.
   *
   * @phpstan-param array{
   *   sets: array<
   *     array{
   *       queue: array{name: string, class: class-string},
   *       operations: array<array{array<mixed>, array<mixed>}>,
   *       finished: array<mixed>|class-string,
   *     }
   *   >
   * } $batch
   */
  public function batchAlter(array &$batch): void {
    // Make up a unique identifier for the set.
    $name = \sprintf('batch_messenger--%s', $this->uuid->generate());

    foreach ($batch['sets'] as &$set) {
      $set['queue'] = [
        // Name is passed to first arg of constructor.
        // database service is always passed to second arg of constructor.
        'name' => $name,
        'class' => BatchMessengerQueue::class,
      ];

      foreach ($set['operations'] as $k => &$operation) {
        $identifier = (string) $k;
        if ($identifier === '') {
          throw new \LogicException('Unexpected $identifier');
        }
        $operation[1][] = BatchMessengerItemIdentifier::create($identifier);
      }

      /** @var string|array{string, string}|null $finishedCallback */
      $finishedCallback = $set['finished'] ?? NULL;
      // Only FinishBatch can call the callback.
      unset($set['finished']);
      if (\is_array($finishedCallback)) {
        $finishedCallback = \implode('::', $finishedCallback);
      }

      $this->batchContextManager->addBatch($name, $finishedCallback);
    }
  }

}
