<?php

declare(strict_types=1);

namespace Drupal\batch_messenger;

use Drupal\batch_messenger\Collection\Collection;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Safe\DateTimeImmutable;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * @internal
 *   For internal use only. When core gets enough API surface to use, this
 *   service will be removed without notice.
 */
final class BatchMessengerTracker {

  /**
   * Memoization of whether a collection was seen during the request.
   *
   * @phpstan-var array<string, mixed>
   */
  private array $seenCollections = [];

  public function __construct(
    private Connection $database,
    #[Autowire(service: 'keyvalue')]
    private KeyValueFactoryInterface $keyValueFactory,
    private TimeInterface $time,
  ) {
  }

  /**
   * @phpstan-param non-empty-string $collection
   * @phpstan-param non-empty-string $identifier
   */
  public function addPending(string $collection, string $identifier): void {
    if (\strlen($collection) > 64 || \strlen($identifier) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    // Register when a collection is seen for the first time.
    $this->seenCollections[$collection] ??= ((function (string $collection): true {
      $this->keyValueFactory
        ->get('batch_messenger-batch')
        ->setIfNotExists($collection, $this->time->getCurrentMicroTime());

      // Value isn't important, so long as it isn't nullable.
      return TRUE;
    })($collection));

    $this->database
      ->insert('batch_messenger__pending')
      ->fields([
        'collection' => $collection,
        'identifier' => $identifier,
      ])
      ->execute();
  }

  /**
   * @phpstan-param non-empty-string $collection
   * @phpstan-param non-empty-string $identifier
   */
  public function transferPendingToProcessed(string $collection, string $identifier): void {
    $this->removePending($collection, $identifier);
    $this->addProcessed($collection, $identifier);
  }

  /**
   * @phpstan-param non-empty-string $collection
   * @phpstan-param non-empty-string $identifier
   */
  private function removePending(string $collection, string $identifier): void {
    $affectedRows = (int) $this->database
      ->delete('batch_messenger__pending')
      ->condition('collection', $collection)
      ->condition('identifier', $identifier)
      ->execute();
    if ($affectedRows === 0) {
      throw new \LogicException(\sprintf('No pending found for %s:%s', $collection, $identifier));
    }
  }

  /**
   * @phpstan-param non-empty-string $collection
   * @phpstan-param non-empty-string $identifier
   */
  private function addProcessed(string $collection, string $identifier): void {
    if (\strlen($collection) > 64 || \strlen($identifier) > 64) {
      throw new \InvalidArgumentException('Length exceeds allowed constraints.');
    }

    $this->database
      ->insert('batch_messenger__processed')
      ->fields([
        'collection' => $collection,
        'identifier' => $identifier,
        'processedOn' => $this->time->getCurrentMicroTime(),
      ])
      ->execute();
  }

  /**
   * Get pending and processed counts for a collection.
   *
   * @phpstan-return array{int<0, max>, int<0, max>}
   */
  public function countItems(string $collection): array {
    // Efficiently get counts of both tables in one transaction.
    $query = $this->database->query(<<<SQL
      SELECT
        (SELECT COUNT(*) FROM {batch_messenger__pending} WHERE collection = :collection) AS pending_count,
        (SELECT COUNT(*) FROM {batch_messenger__processed} WHERE collection = :collection) AS processed_count
      SQL, [':collection' => $collection]) ?? throw new \Exception('Bad query');

    /** @var array{pending_count: string, processed_count: string} $result */
    $result = $query->fetchAssoc();

    // @phpstan-ignore-next-line
    return [(int) $result['pending_count'], (int) $result['processed_count']];
  }

  /**
   * @phpstan-return \Generator<Collection>
   */
  public function getCollections(): \Generator {
    /** @var non-empty-string $collectionName */
    foreach ($this->keyValueFactory->get('batch_messenger-batch')->getAll() as $collectionName => $created) {
      // @todo push this as a lazy callable to the Collection object.
      [$pendingCount, $processedCount] = $this->countItems($collectionName);
      yield Collection::create(
        $collectionName,
        DateTimeImmutable::createFromFormat('U', (string) (int) $created),
        $pendingCount,
        $processedCount,
      );
    }
  }

  public function clearCompleteCollections(): void {
    foreach ($this->getCollections() as $collection) {
      // @todo pad this out with deletes for the other tables and KV.
      if ($collection->isComplete()) {
        $this->keyValueFactory->get('batch_messenger-batch')->delete($collection->collectionName);
      }
    }
  }

  public function getMostRecentlyUpdatedCollection(): ?Collection {
    $query = $this->database
      ->select('batch_messenger__processed', 'p')
      ->fields('p', ['collection'])
      ->orderBy('processedOn', 'DESC')
      ->range(0, 1);

    /** @var non-empty-string|false $collectionName */
    $collectionName = $query->execute()?->fetchField() ?? throw new \Exception('Bad query');
    if ($collectionName === FALSE) {
      return NULL;
    }

    [$pendingCount, $processedCount] = $this->countItems($collectionName);
    return Collection::create(
      $collectionName,
      // @todo fixme
      new \DateTimeImmutable(),
      $pendingCount,
      $processedCount,
    );
  }

}
