<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\Ui\Controller;

use Drupal\batch_messenger\BatchBridge\BatchMessengerBatchContextManager;
use Drupal\batch_messenger\BatchMessengerTracker;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;

final class CollectionController extends ControllerBase {

  public function __construct(
    private BatchMessengerTracker $batchMessengerTracker,
    private DateFormatterInterface $dateFormatter,
    private ?BatchMessengerBatchContextManager $batchContextManager = NULL,
  ) {}

  /**
   * @phpstan-return array<string, mixed>
   */
  public function __invoke(): array {
    $rows = [];
    foreach ($this->batchMessengerTracker->getCollections() as $collection) {
      $rows[] = [
        $collection->collectionName,
        $this->dateFormatter->format($collection->created->getTimestamp()),
        $collection->pendingCount,
        $collection->processedCount,
        [
          'data' => [
            '#type' => 'inline_template',
            '#template' => '<span class="gin-status">{{ status }}</span>',
            '#context' => [
              'status' => $collection->isComplete() ? $this->t('Complete') : $this->t('Incomplete'),
            ],
          ],
        ],
      ];

      if (!$collection->isComplete()) {
        $rows[] = [
          [
            'colspan' => 5,
            'data' => [
              '#theme' => 'progress_bar',
              '#percent' => \round($collection->progress() * 100, 3),
              '#message' => $this->batchContextManager?->getLatestMessage($collection->collectionName),
            ],
          ],
        ];
      }
    }

    $build = [];
    $build['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Set'),
        $this->t('Created on'),
        $this->t('Pending'),
        $this->t('Processed'),
        $this->t('Status'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No message collections found.'),
    ];

    // We don't use cache tags, and its more useful to never cache this info.
    (new CacheableMetadata())
      ->setCacheMaxAge(0)
      ->applyTo($build);

    return $build;
  }

}
