<?php

declare(strict_types=1);

namespace Drupal\batch_messenger\Ui;

use Drupal\batch_messenger\BatchMessengerTracker;
use Drupal\batch_messenger\Ui\EventListener\RouteSubscriber;
use Drupal\Core\Menu\LocalActionDefault;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;

final class UiHooks {

  public function __construct(
    private BatchMessengerTracker $batchMessengerTracker,
  ) {
  }

  /**
   * Implements hook_menu_links_discovered_alter().
   *
   * @phpstan-param array<mixed> $links
   */
  public function menuLinksDiscoveredAlter(&$links): void {
    $links['batch_messenger.ui.collection'] = [
      'title' => new TranslatableMarkup('Message sets'),
      'route_name' => RouteSubscriber::COLLECTION_ROUTE_NAME,
      'description' => new TranslatableMarkup('Show batches of messages.'),
      'parent' => 'system.admin_reports',
    ];
  }

  /**
   * Implements hook_toolbar().
   *
   * @todo convert to a Hooks class conditionally added by the parameter.
   *
   * @phpstan-return array<string, mixed>
   */
  public function toolbar(): array {
    $items = [];

    $collection = $this->batchMessengerTracker->getMostRecentlyUpdatedCollection();

    $items['batch_messenger__progress'] = [
      '#type' => 'toolbar_item',
      '#weight' => -1000,
      '#cache' => [
        // @todo cache longer when nothing inprogress, shorter when inprogress.
        'max-age' => 1,
      ],
    ];

    if ($collection?->isComplete() === FALSE) {
      $items['batch_messenger__progress']['tab'] = [
        '#prefix' => \sprintf(
          '<div class="batch_messenger-toolbar-item"><a href="%s">',
          Url::fromRoute(RouteSubscriber::COLLECTION_ROUTE_NAME)->toString(),
        ),
        '#theme' => 'progress_bar',
        '#percent' => $collection->progress() * 100,
        '#suffix' => '</a></div>',
        '#attached' => [
          'library' => [
            'batch_messenger/toolbar',
          ],
        ],
      ];
    }

    return $items;
  }

  /**
   * Implements hook_menu_local_actions_alter().
   *
   * @phpstan-param array<string, mixed> $local_actions
   */
  public function menuLocalActionsAlter(&$local_actions): void {
    $local_actions['batch_messenger.ui.clear_complete'] = [
      'title' => \t('Clear complete'),
      'route_name' => RouteSubscriber::CLEAR_COMPLETE_ROUTE_NAME,
      'appears_on' => [
        RouteSubscriber::COLLECTION_ROUTE_NAME,
      ],
      'class' => LocalActionDefault::class,
      'weight' => 100,
      'options' => [
        'attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ],
    ];
  }

  /**
   * Implements hook_library_info_build().
   *
   * @phpstan-return array<string, mixed>
   */
  public function libraryInfoBuild(): array {
    $libraries = [];

    $libraries['toolbar'] = [
      'css' => [
        'component' => [
          'css/Ui/toolbar.css' => [],
        ],
      ],
    ];

    return $libraries;
  }

}
