<?php

declare(strict_types=1);

namespace Drupal\Tests\batch_messenger\Kernel;

use Drupal\batch_messenger\BatchMessengerTracker;
use Drupal\batch_messenger\Messenger\Stamp\CollectionItem;
use Drupal\batch_messenger_test\TestMessage;
use Drupal\batch_messenger_test\TimeMachine;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\sm\Traits\SmAsyncTrait;
use Drupal\Tests\sm\Traits\SmMappingTrait;
use Symfony\Component\Clock\MockClock;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Messenger\Worker;

/**
 * @coversDefaultClass \Drupal\batch_messenger\BatchMessengerTracker
 */
final class BatchMessengerTest extends KernelTestBase {

  use SmAsyncTrait;
  use SmMappingTrait;

  protected static $modules = [
    'sm',
    'batch_messenger',
    'batch_messenger_test',
  ];

  private string $currentTime = '9am 10 Jan 2014 UTC';

  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('batch_messenger', [
      'batch_messenger__pending',
      'batch_messenger__processed',
      'batch_messenger__batch',
      'batch_messenger__batch_messages',
    ]);
  }

  public function testTracker(): void {
    $worker = static::worker();

    $message = new TestMessage();

    $this->setRouting([
      $message::class => 'sm.transport.asynchronous',
    ]);

    static::assertEquals([], \iterator_to_array(static::batchMessengerTracker()->getCollections()));

    /** @var non-empty-string $collectionName */
    $collectionName = $this->randomMachineName();
    static::bus()->dispatch(new Envelope($message, [CollectionItem::create($collectionName, '#id1')]));

    static::assertEquals([1, 0], static::batchMessengerTracker()->countItems($collectionName));

    $collections = \iterator_to_array(static::batchMessengerTracker()->getCollections());
    self::assertCount(1, $collections);
    static::assertEquals($collectionName, $collections[0]->collectionName);
    static::assertEquals('1389344400.000000', $collections[0]->created->format('U.u'));

    $worker();
    static::assertEquals([0, 1], static::batchMessengerTracker()->countItems($collectionName));
    static::assertEquals([], static::dumpRows('batch_messenger__pending'));
    static::assertEquals([
      [
        'collection' => $collectionName,
        'identifier' => '#id1',
        'processedOn' => '1389344400.000000',
      ],
    ], static::dumpRows('batch_messenger__processed'));
    static::assertEquals([], static::dumpRows('batch_messenger__pending'));
    static::assertEquals([], static::dumpRows('batch_messenger__batch'));
    static::assertEquals([], static::dumpRows('batch_messenger__batch_messages'));
  }

  private static function bus(): MessageBusInterface {
    /** @var \Symfony\Component\Messenger\MessageBusInterface */
    return \Drupal::service(MessageBusInterface::class);
  }

  private static function batchMessengerTracker(): BatchMessengerTracker {
    return \Drupal::service(BatchMessengerTracker::class);
  }

  private function worker(): callable {
    return new class ($this->createWorker()) {

      public function __construct(private Worker $worker) {}

      public function __invoke(): void {
        $this->worker->run();
      }

    };
  }

  private function createWorker(): Worker {
    return new Worker(
      receivers: ['asynchronous' => $this->receiver()],
      bus: static::bus(),
      eventDispatcher: $this->dispatcher(maximumNumberOfMessages: 1),
      clock: new MockClock($this->currentTime),
    );
  }

  /**
   * @phpstan-return array<int, array<mixed>>
   */
  private function dumpRows(string $tableName): array {
    return \Drupal::database()
      ->select($tableName, 't')
      ->fields('t')
      ->execute()
      ?->fetchAll(\PDO::FETCH_ASSOC) ?? throw new \Exception('Bad query');
  }

  public function register(ContainerBuilder $container): void {
    parent::register($container);

    $this->setSmRoutingToContainer($container);

    $container->getDefinition('datetime.time')
      ->setClass(TimeMachine::class)
      ->setArgument(0, $this->currentTime)
      ->setArgument(1, 0)
      ->setArgument(2, TRUE);

    // This test should work without UI and BatchBridge.
    $container->setParameter('batch_messenger.batch_bridge', FALSE);
    $container->setParameter('batch_messenger.ui', FALSE);
  }

}
