<?php

namespace Drupal\user_importer\Batch;

use Drupal\user\Entity\User;

final class UserImportBatch {
  public static function processRow(array $row, array &$context): void {
    $context['results']['processed'] = ($context['results']['processed'] ?? 0) + 1;

    $name  = trim((string) ($row['username'] ?? ''));
    $mail  = trim((string) ($row['email'] ?? ''));
    $pass  = (string) ($row['password'] ?? '');
    $roles = array_filter(array_map('trim', explode(',', (string) ($row['roles'] ?? ''))));
    $status = (string) ($row['status'] ?? '');
    $active = $status === '' ? 1 : (in_array(strtolower($status), ['1','active','enabled','true','yes'], true) ? 1 : 0);

    if ($name === '' || !filter_var($mail, FILTER_VALIDATE_EMAIL)) {
      $context['results']['skipped'][] = ['row' => $row, 'reason' => 'Invalid username/email'];
      return;
    }

    $uids = \Drupal::entityQuery('user')->condition('mail', $mail)->execute();
    $account = $uids ? User::load(reset($uids)) : User::create();

    if ($account->isNew()) {
      $account->setUsername($name);
      $account->setEmail($mail);
    } else {
      if ($account->getEmail() !== $mail) {
        $account->setEmail($mail);
      }
    }

    if ($pass !== '') {
      $account->setPassword($pass);
    }
    $account->set('status', $active);

    $allRoles = array_keys(\Drupal::service('entity_type.manager')->getStorage('user_role')->loadMultiple());
    foreach ($roles as $r) {
      $machine = strtolower(preg_replace('/\s+/', '_', $r));
      if ($machine !== 'authenticated' && in_array($machine, $allRoles, true)) {
        $account->addRole($machine);
      }
    }

    try {
      $account->save();
    } catch (\Throwable $e) {
      $context['results']['failed'][] = ['row' => $row, 'error' => $e->getMessage()];
    }
  }

  public static function finish(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();
    if (!$success) {
      $messenger->addError(t('User import did not complete.'));
      return;
    }
    $processed = $results['processed'] ?? 0;
    $messenger->addStatus(t('Processed: @p', ['@p' => $processed]));
  }
}
