<?php

namespace Drupal\user_importer\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\user_importer\Service\ExcelReader;
use Drupal\user_importer\Batch\UserImportBatch;

final class UserImportForm extends FormBase {

  // Make it nullable and init later.
  private ?ExcelReader $reader = null;

  public function getFormId(): string { return 'user_importer_form'; }

  private function excelReader(): ExcelReader {
    // Lazy load in case the container factory wasn’t used.
    return $this->reader ??= \Drupal::service('user_importer.excel_reader');
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Excel file (.xlsx)'),
      '#required' => TRUE,
      '#upload_location' => 'private://user_import/',
      '#upload_validators' => [
        'file_validate_extensions' => ['xlsx xls'],
        'file_validate_size' => [20 * 1024 * 1024],
      ],
    ];
    $form['header_row'] = [
      '#type' => 'number',
      '#title' => $this->t('Header row number'),
      '#default_value' => 1,
      '#min' => 1,
      '#required' => TRUE,
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Start import'),
      '#button_type' => 'primary',
    ];
    return $form;
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $fids = (array) $form_state->getValue('file');
    if (!$fids) {
      $form_state->setErrorByName('file', $this->t('File is required.'));
      return;
    }
    $file = File::load(reset($fids));
    if (!$file) {
      $form_state->setErrorByName('file', $this->t('Cannot load the uploaded file.'));
      return;
    }
    $real = \Drupal::service('file_system')->realpath($file->getFileUri());
    if (!$real || !is_file($real)) {
      $form_state->setErrorByName('file', $this->t('Uploaded file is missing on disk.'));
    }
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $fids = (array) $form_state->getValue('file');
    $file = File::load(reset($fids));
    if (!$file) {
      $this->messenger()->addError($this->t('Cannot load the uploaded file.'));
      return;
    }

    $file->setPermanent();
    $file->save();
    \Drupal::service('file.usage')->add($file, 'user_importer', 'file', $file->id());

    $path = \Drupal::service('file_system')->realpath($file->getFileUri());
    $headerRow = (int) $form_state->getValue('header_row');
    $rows = $this->excelReader()->readRows($path, $headerRow);

    if (!$rows) {
      $this->messenger()->addWarning($this->t('No data rows found.'));
      return;
    }

    $operations = [];
    foreach ($rows as $row) {
      $operations[] = [[UserImportBatch::class, 'processRow'], [$row]];
    }

    $batch = [
      'title' => $this->t('Importing users'),
      'operations' => $operations,
      'finished' => [UserImportBatch::class, 'finish'],
      'progress_message' => $this->t('Processed @current of @total.'),
      'error_message' => $this->t('An error occurred during import.'),
    ];
    batch_set($batch);
  }
}