<?php

namespace Drupal\bee_hotel\Form;

use Drupal\beehotel_utils\BeeHotelCommerce;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure bee_hotel settings.
 */
class BeeHotelSettingsForm extends ConfigFormBase {

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'bee_hotel.settings';

  /**
   * Drupal configuration service container.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected $configFactory;

  /**
   * The BeeHotel commerce Util.
   *
   * @var \Drupal\beehotel_utils\BeeHotelCommerce
   */
  protected $beehotelCommerce;

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory.
   * @param \Drupal\beehotel_utils\BeeHotelCommerce $beehotel_commerce
   *   BeeHotel Commerce Utils.
   */
  public function __construct(ConfigFactory $config_factory, BeeHotelCommerce $beehotel_commerce) {
    $this->configFactory = $config_factory;
    $this->beehotelCommerce = $beehotel_commerce;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('beehotel_utils.beehotelcommerce'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'beehotel_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $config = $this->config(static::SETTINGS);

    $path = \Drupal::service('extension.list.module')->getPath('bee_hotel');

    // Vertical Tabs.
    $form['settings'] = [
      '#type' => 'vertical_tabs',
      '#default_tab' => 'edit-publication',
    ];

    // A. Main.
    $form['main'] = [
      '#type' => 'details',
      '#title' => $this->t('Main'),
      '#group' => 'settings',
    ];

    $form['main']['off'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Reservations OFF'),
      '#collapsible' => FALSE,
    ];

    $form['main']['off']['off_value'] = [
      '#default_value' => $config->get('beehotel.off_value'),
      '#type' => 'checkbox',
      '#title' => $this->t('Switch the reservation system OFF'),
      '#description' => $this->t('When OFF, user booking units will receive a message saying the reservation system is OFF'),
    ];

    $form['main']['off']['off_text'] = [
      '#default_value' => $config->get('beehotel.off_text'),
      '#type' => 'textfield',
      '#title' => $this->t('A message to guests, saying reservation system is OFF.'),
      '#description' => $this->t('When OFF, user booking units will receive this message.'),
    ];

    $form['main']['setupmode'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('SETUP mode'),
      '#collapsible' => FALSE,
    ];

    $form['main']['setupmode']['setupmode_on_value'] = [
      '#default_value' => $config->get('beehotel.setup_mode'),
      '#type' => 'checkbox',
      '#title' => $this->t('Switch the SETUP mode ON'),
      '#description' => $this->t('When ON, a deep diagnostic check is made. With SETUP mode ON, your system maybe slower. Disable this once your BEE Hotel enviroment is working fine'),
    ];

    $weight = [
      'bee' => bee_hotel_get_module_weight('bee'),
      'bee_hotel' => bee_hotel_get_module_weight('bee_hotel'),
    ];

    $details = "<div>";
    $details .= "Bee HOTEL weight: " . $weight['bee_hotel'] . "<br/>";
    $details .= "BEE weight: " . $weight['bee'] . "<br/>";

    if ($weight['bee'] <= $weight['bee_hotel']) {
      $weight['status'] = $this->t("Please update bee_hotel weight");
    }
    else {
      $weight['status'] = $this->t("Modules weight look good!");
    }

    // Book this unit.
    $details .= $weight['status'];
    $details .= "</div>";

    $form['main']['weight'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Module weight'),
      '#description' => $this->t('Bee Hotel weight matters. Check values below'),
      '#collapsible' => FALSE,
    ];

    $form['main']['weight']['update'] = [
      '#prefix' => $details,
      '#type' => 'submit',
      '#value' => $this->t('Update weight'),
      '#submit' => ['::submitUpdateweight'],
    ];

    // B. Booking.
    $form['booking'] = [
      '#type' => 'details',
      '#title' => $this->t('Booking'),
      '#group' => 'settings',
    ];

    $form['booking']['calendar'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Calendar'),
      '#collapsible' => FALSE,
    ];

    $form['booking']['calendar']['calendar_from'] = [
      '#default_value' => $config->get('beehotel.calendar_from'),
      '#type' => 'select',
      '#options' => [
        0 => $this->t('Today'),
        1 => $this->t('Tomorrow'),
        2 => $this->t('2 days'),
      ],
      '#title' => $this->t('Accept reservations from'),
      '#description' => $this->t('Selecting today, you should be ready to checkin Guest with few hours time. This feature is currently @TODO'),
    ];

    // Cancellation Policy.
    $form['booking']['cancellation_policy'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Cancellation Policy'),
      '#collapsible' => FALSE,
    ];

    // Build cancellation policy days options.
    $cancellation_options = [0 => $this->t('No cancellation policy')];

    // Add days from 1 to 30.
    for ($i = 1; $i <= 30; $i++) {
      $cancellation_options[$i] = $i;
    }

    // Add additional specific days.
    $additional_days = [40, 45, 50, 60];
    foreach ($additional_days as $day) {
      $cancellation_options[$day] = $day;
    }

    $form['booking']['cancellation_policy']['cancellation_policy_days'] = [
      '#type' => 'select',
      '#title' => $this->t('Cancellation Policy Days'),
      '#description' => $this->t('Number of days before arrival within which guests can cancel their reservation and receive a 100% refund.'),
      '#options' => $cancellation_options,
      '#default_value' => $config->get('beehotel.cancellation_policy_days') ?? 0,
      '#required' => TRUE,
    ];

    $form['booking']['booking_forms'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Booking Forms'),
      '#description' => $this->t("Custom settings for booking forms"),
      '#collapsible' => FALSE,
    ];

    $description = [];

    $description['pieces'][] = $this->t("The 'Units Search' form allows Guests to search for units with given dates and occupants.");
    $description['pieces'][] = $this->t("Options:");
    $description['pieces'][] = $this->t('<code><<i>none</i>></code> : nothing');
    $description['pieces'][] = $this->t('<code><<i>ct-label</i>></code> : the Content type label');
    $description['pieces'][] = $this->t('<title><<i>title</i>></code> : the node Title');
    $description['pieces'][] = $this->t("A 'String': the 'String' itself");
    $description['output'] = implode("<br/>", $description['pieces']);

    // Units search.
    $form['booking']['booking_forms']['units_search'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Units Search'),
      '#description' => $description['output'],
      '#collapsible' => FALSE,
    ];

    $form['booking']['booking_forms']['units_search']['units_search_position'] = [
      '#default_value' => $config->get('beehotel.units_search_position'),
      '#options' => [
        'none' => $this->t("none"),
        'top' => $this->t("Top"),
        'bottom' => $this->t("Bottom"),
      ],
      '#type' => 'select',
      '#title' => $this->t('Position'),
      '#required' => TRUE,
      '#description' => $this->t('Position inside the node display. @todo: code for the top and bottom position'),
    ];

    $form['booking']['booking_forms']['units_search']['units_search_submit_label'] = [
      '#default_value' => $config->get('beehotel.units_search_submit'),
      '#type' => 'textfield',
      '#title' => $this->t('Submit label'),
      '#required' => TRUE,
      '#description' => $this->t('The label of the submit button'),
    ];

    $form['booking']['booking_forms']['units_search']['units_search_header_label'] = [
      '#default_value' => $config->get('beehotel.units_search_header'),
      '#type' => 'textfield',
      '#title' => $this->t('Header label'),
      '#description' => $this->t('A string introducing the form'),
    ];

    $description = [];

    $description['pieces'][] = $this->t("The 'Book this unit' form allows guests to book a given BeeHotelUnit");
    $description['pieces'][] = $this->t("The Form is exposed in the node view of the Unit");
    $description['pieces'][] = $this->t("You can here decide how this Form is introducted to the audience");
    $description['pieces'][] = $this->t("Options:");
    $description['pieces'][] = $this->t('<code><<i>none</i>></code> : nothing');
    $description['pieces'][] = $this->t('<code><<i>ct-label</i>></code> : the Content type label');
    $description['pieces'][] = $this->t('<title><<i>title</i>></code> : the node Title');
    $description['pieces'][] = $this->t("A 'String': the 'String' itself");
    $description['output'] = implode("<br/>", $description['pieces']);

    $form['booking']['booking_forms']['book_this_unit'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Book this unit Form'),
      '#description' => $description['output'],
      '#collapsible' => FALSE,
    ];

    $form['booking']['booking_forms']['book_this_unit']['book_this_unit_position'] = [
      '#default_value' => $config->get('beehotel.book_this_unit_position'),
      '#options' => [
        'none' => $this->t("none"),
        'top' => $this->t("Top"),
        'bottom' => $this->t("Bottom"),
      ],
      '#type' => 'select',
      '#title' => $this->t('Position'),
      '#required' => TRUE,
      '#description' => $this->t('Position inside the node display. @todo: code for the top and bottom position'),
    ];

    $form['booking']['booking_forms']['book_this_unit']['book_this_unit_submit_label'] = [
      '#default_value' => $config->get('beehotel.book_this_unit_submit'),
      '#type' => 'textfield',
      '#title' => $this->t('Submit label'),
      '#required' => TRUE,
      '#description' => $this->t('The label of the submit button'),
    ];

    $form['booking']['booking_forms']['book_this_unit']['book_this_unit_header_label'] = [
      '#default_value' => $config->get('beehotel.book_this_unit_header'),
      '#type' => 'textfield',
      '#title' => $this->t('Header label'),
      '#description' => $this->t('A string introducing the form'),
    ];

    // Vertical.
    $form['vertical'] = [
      '#type' => 'details',
      '#title' => $this->t('Vertical'),
      '#group' => 'settings',
    ];

    $form['vertical']['setup'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => "Set up <a href='/admin/beehotel/vertical/settings'>Setup vertical</a>",
      '#title' => $this->t('Set up Vertical'),
    ];

    // Price Alteration.
    $form['pricelateration'] = [
      '#type' => 'details',
      '#title' => $this->t('Price Alteration'),
      '#group' => 'settings',
    ];

    $form['pricelateration']['chain_chart'] = [
      '#type' => 'radios',
      '#title' => $this->t('Chain chart'),
      '#default_value' => $config->get('beehotel.chain_chart'),
      '#options' => [
        'pie' => $this->t('Pie') . "<img src='/" . $path . "/assets/images/google_chart_pie.jpg' alt='Google pie chart' height=40 width=40>",
        'combochart' => $this->t('Combo Chart') . "<img src='/" . $path . "/assets/images/google_chart_combo.jpg' alt='Google combo chart' height=40 width=120>",
      ],
      '#description' => $this->t('This feature is ALPHA'),
    ];

    $form['pricelateration']['setuppage'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => "Set up <a href='/admin/beehotel/pricealterator/alterators'>price alterators</a>",
      '#title' => $this->t('Setup Alter ators'),
    ];

    // Guest Messages.
    $form['guestmessages'] = [
      '#type' => 'details',
      '#title' => $this->t('Guest Messages'),
      '#group' => 'settings',
      '#description' => $this->t('Customize your Guest Messages'),
    ];

    $form['guestmessages']['tokens'] = [
      '#type' => 'details',
      '#title' => $this->t('Tokens'),
      '#description' => $this->t('Customize tokens into Guest messages'),
    ];

    $form['guestmessages']['tokens']['balance_cash_euro'] = [
      '#type' => 'details',
      '#title' => $this->t('Guest Messages'),
      '#group' => 'settings',
      '#description' => $this->t('Customize your Guest Messages'),
    ];

    $form['guestmessages']['tokens']['balance_cash_euro']['balance_cash_subtract'] = [
      '#default_value' => $config->get('beehotel.guestmessages.balance_cash_euro_subtract'),
      '#type' => 'textfield',
      '#title' => $this->t('@symbol to subtracted for cash Euro balance', ['@symbol' => $this->beehotelCommerce->currentStoreCurrency()->get('symbol')]),
    ];

    $form['guestmessages']['tokens']['balance_cash_euro']['smartceiling'] = [
      '#default_value' => $config->get('beehotel.guestmessages.smartceiling'),
      '#type' => 'checkbox',
      '#title' => $this->t('Smart Ceiling'),
      '#description' => $this->t('Balance will be ceiled'),
    ];

    $form['dateandtime'] = [
      '#type' => 'details',
      '#title' => $this->t('Date and time'),
      '#group' => 'settings',
      '#description' => $this->t('This values will be used for scheduling and communications across the platform'),
    ];

    // Generate time options with 30-minute increments.
    $time_options = [];
    for ($hour = 0; $hour < 24; $hour++) {
      for ($minute = 0; $minute < 60; $minute += 30) {
        $time_value = sprintf('%02d:%02d', $hour, $minute);
        $time_display = \Drupal::service('date.formatter')->format(strtotime($time_value), 'custom', 'g:i A');
        $time_options[$time_value] = $time_display;
      }
    }

    $form['dateandtime']['default_checkin_time'] = [
      '#type' => 'select',
      '#title' => $this->t('Default check-in time'),
      '#description' => $this->t('The default time for check-in.'),
      '#options' => $time_options,
      '#default_value' => $config->get('beehotel.dateandtime.default_checkin_time') ?: '14:00',
      '#empty_option' => $this->t('- Select time -'),
    ];

    $form['dateandtime']['default_checkout_time'] = [
      '#type' => 'select',
      '#title' => $this->t('Default check-out time'),
      '#description' => $this->t('The default time for check-out.'),
      '#options' => $time_options,
      '#default_value' => $config->get('beehotel.dateandtime.default_checkout_time') ?: '11:00',
      '#empty_option' => $this->t('- Select time -'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate cancellation policy days.
    $cancellation_days = $form_state->getValue('cancellation_policy_days');
    $valid_days = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 40, 45, 50, 60];

    if (!in_array($cancellation_days, $valid_days)) {
      $form_state->setErrorByName('cancellation_policy_days', $this->t('The cancellation policy days value is not valid. Please select a value from the available options.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $config = $this->configFactory->getEditable(static::SETTINGS);
    $config
      ->set('beehotel.off_value', $form_state->getValue('off_value'))
      ->set('beehotel.off_text', $form_state->getValue('off_text'))
      ->set('beehotel.setup_mode', $form_state->getValue('setupmode_on_value'))
      ->set('beehotel.calendar_from', $form_state->getValue('calendar_from'))
      ->set('beehotel.cancellation_policy_days', $form_state->getValue('cancellation_policy_days'))
      ->set('beehotel.book_this_unit_header', $form_state->getValue('book_this_unit_header_label'))
      ->set('beehotel.book_this_unit_submit', $form_state->getValue('book_this_unit_submit_label'))
      ->set('beehotel.book_this_unit_position', $form_state->getValue('book_this_unit_position'))
      ->set('beehotel.units_search_header', $form_state->getValue('units_search_header_label'))
      ->set('beehotel.units_search_submit', $form_state->getValue('units_search_submit_label'))
      ->set('beehotel.units_search_position', $form_state->getValue('units_search_position'))
      ->set('beehotel.chain_chart', $form_state->getValue('chain_chart'))
      ->set('beehotel.guestmessages.balance_cash_subtract', $form_state->getValue('balance_cash_subtract'))
      ->set('beehotel.guestmessages.smartceiling', $form_state->getValue('smartceiling'))
      ->set('beehotel.dateandtime.default_checkin_time', $form_state->getValue('default_checkin_time'))
      ->set('beehotel.dateandtime.default_checkout_time', $form_state->getValue('default_checkout_time'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Update modules weight.
   */
  public function submitUpdateweight(array &$form, FormStateInterface $form_state) {
    bee_hotel_update_modules_weight();
    $this->messenger()->addStatus($this->t('Modules weight updated'));
  }

}
