<?php

namespace Drupal\better_taxonomy;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityListBuilderInterface;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\HttpFoundation\RequestStack;

class BetterTaxonomyService {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The request stack.
   *
   * @var RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The term list builder.
   */
  protected EntityListBuilderInterface $termListBuilder;

  /**
   * Class constructor.
   *
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager,
                              RequestStack $requestStack) {
    $this->entityTypeManager = $entityTypeManager;
    $this->requestStack = $requestStack;
    $this->termListBuilder = $entityTypeManager->getListBuilder('taxonomy_term');

  }

  /**
   * Returns "search" query parameter or FALSE.
   *
   * @return bool|string
   *   "search" query parameter or FALSE.
   */
  public function CheckSearch(): bool|string {
    $request = $this->requestStack->getCurrentRequest();
    $search_parameter = $request->query->get('search');
    if (empty($search_parameter)) {
      return FALSE;
    }
    return $search_parameter;
  }

  /**
   * Given a string, returns all terms of vocabulary containing this string.
   *
   * @param string $search_parameter
   *   String to search.
   * @param string $vid
   *   Vocabulary id.
   *
   * @return array
   *   List of terms.
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function GetSearchList(string $search_parameter, string $vid): array {
    $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $query = $term_storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('vid', $vid)
      ->condition('name', '%' .$search_parameter . '%', 'LIKE')
      ->sort('name', 'ASC');
    $term_ids = $query->execute();
    return $term_storage->loadMultiple($term_ids);
  }

  /**
   * Creates search result table.
   *
   * @param string $vocabulary
   *   Vocabulary id.
   * @param string $search_parameter
   *   String to search.
   * @param array $terms
   *   Search results.
   *
   * @return array
   *  Table.
   *
   * @throws EntityMalformedException
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function GetSearchResults(string $vocabulary, string $search_parameter, array $terms): array {
    $table = [
      '#type' => 'table',
      '#empty' => $this->t('No existing terms containing "@search".', ['@search' => $search_parameter]),
      '#header' => [
        'term' => $this->t('Name'),
        'level' => $this->t('Level'),
        'parent' => $this->t('Parent'),
        'status' => $this->t('Status'),
        'operations' => $this->t('Operations'),
      ],
      '#attributes' => [
        'id' => 'taxonomy',
      ],
    ];
    if (!empty($terms)) {
      // Retrieve vocabulary max depth.
      $max_depth = $this->getVocabularyMaxDepth($vocabulary);
      foreach ($terms as $key => $term) {
        $table[$key] = $this->getTableRow($term, $max_depth);
      }
      if ($max_depth == 0) {
        unset($table['#header']['parent']);
      }
      if ($max_depth <= 1) {
        unset($table['#header']['level']);
      }
    }
    return $table;
  }

  /**
   * Creates search result table row.
   *
   * @param Term $term
   *   Term.
   * @param int $max_depth
   *   Max depth of this vocabulary.
   *
   * @return array[]
   *   Table row.
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   * @throws EntityMalformedException
   */
  private function getTableRow(Term $term, int $max_depth): array {
    $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $level = [];
    if ($max_depth > 0) {
      $parents = $term_storage->loadParents($term->id());
      $parent = reset($parents);
      if (!empty($parent) && $max_depth > 1) {
        $tree = $term_storage->loadTree($term->bundle(), 0, NULL, FALSE);
        $filter_by = $term->id();
        $current_term = array_filter($tree, function($item) use ($filter_by) {
          return ($item->tid == $filter_by);
        });
        $current_term = reset($current_term);
        if (isset($current_term->depth) && $current_term->depth > 0) {
          $level = $current_term->depth;
        }
      }
    }
    $operations = [];
    if ($operations_list = $this->termListBuilder->getOperations($term)) {
      $operations = [
        '#type' => 'operations',
        '#links' => $operations_list,
      ];
    }
    $row = [
      'term' => [
        '#type' => 'link',
        '#title' => $term->getName(),
        '#url' => $term->toUrl(),
      ],
      'level' => [
        '#type' => 'item',
        '#markup' => !empty($level) ? $level : 0,
      ],
      'parent' => [
        '#type' => 'item',
        '#markup' => !empty($parent) ? $parent->getName() : '--',
      ],
      'status' => [
        '#type' => 'item',
        '#markup' => ($term->isPublished()) ? $this->t('Published') : $this->t('Unpublished'),
      ],
      'operations' => $operations,
      '#term' => $term,
    ];
    if ($max_depth == 0) {
      unset($row['parent']);
    }
    if ($max_depth <= 1) {
      unset($row['level']);
    }
    return $row;
  }

  /**
   * Creates the terms respecting the hierarchy.
   *
   * @param $values
   *   Items to create.
   *
   * @return array
   *   All terms created.
   *
   * @throws EntityStorageException
   */
  public function addMultipleTerms($values): array {
    $vocabulary = $values['vocabulary'];
    // Check parent term if any.
    $base_term = $this->getBaseTerm($values);
    // Parse terms multi line string.
    $terms = $this->parseMultipleTerms($values['terms'], $vocabulary);
    // Save terms.
    $saved_terms = [];
    $this->recursiveCreateTerms($saved_terms, $terms, $base_term, $vocabulary);
    return $saved_terms;
  }

  /**
   * Recursive function to create a hierarchy tree of terms.
   *
   * @param array $saved_terms
   *   All terms parsed.
   * @param array $items
   *   Items to parse as array.
   * @param int $base_term
   *   Parent terms of all hierarchy.
   * @param string $vocabulary
   *   Vocabulary id.
   * @param int|null $parent_tid
   *   Parent term id if any.
   *
   * @return void
   *
   * @throws EntityStorageException
   */
  private function recursiveCreateTerms(array &$saved_terms, array $items, int $base_term, string $vocabulary, ?int $parent_tid = NULL): void {
    foreach ($items as $item) {
      // Create the term.
      $term = Term::create([
        'vid' => $vocabulary,
        'name' => $item['name'],
        'parent' => $parent_tid ? [$parent_tid] : $base_term,
      ]);
      $term->save();
      $tid = $term->id();
      $saved_terms[] = $item['name'];

      // Recurse into children.
      if (!empty($item['children'])) {
        $this->recursiveCreateTerms($saved_terms, $item['children'], $base_term, $vocabulary, $tid);
      }
    }
  }

  /**
   * Given a vocabulary, returns max hierarchy depth.
   *
   * @param string $vocabulary
   *   Vocabulary id.
   * @param array $vocabulary_tree
   *   Vocabulary tree.
   *
   * @return int
   *   Vocabulary max depth.
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function getVocabularyMaxDepth(string $vocabulary, array $vocabulary_tree = []): int {
    $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
    if (empty($vocabulary_tree)) {
      $vocabulary_tree = $term_storage->loadTree($vocabulary, 0, NULL, TRUE);
    }
    $tree = [];
    // Parse all vocabulary to get max depth.
    foreach ($vocabulary_tree as $term) {
      $tree[$term->depth][$term->id()] = $term->label();
    }
    return max(array_keys($tree));
  }

  /**
   * Given all select levels, returns de last one
   *
   * @param array $values
   *   All form_State values.
   *
   * @return int
   *   The parent term selected.
   */
  private function getBaseTerm(array $values): int {
    $base = 0;
    $levels = [];
    foreach ($values as $field => $value) {
      if (str_starts_with($field, 'level_')) {
        $key = substr($field, 6);
        $levels[$key] = (int) $value;
      }
    }
    $levels = array_filter($levels);
    if (!empty($levels)) {
      $max_key = max(array_keys($levels));
      $base = $levels[$max_key];
    }
    return $base;
  }

  /**
   * Converts multiline terms to array.
   *
   * @param string $terms
   *   Multiline terms.
   * @param string $vid
   *   Current vocabulary id.
   *
   * @return array
   *   Array of terms to add.
   */
  private function parseMultipleTerms(string $terms, string $vid): array {
    // Convert multiline terms to array.
    $names_array = explode(PHP_EOL, $terms);
    $terms_array = [];
    $terms_max_length = $this->getVocabularyMaxLength($vid);
    // Clean term names.
    foreach ($names_array as $name) {
      $name = trim($name);
      $name = $this->sanitizeName($name);
      if (strlen($name) > $terms_max_length) {
        $name = substr($name, 0, $terms_max_length);
      }
      $terms_array[] = $name;
    }
    $terms_array = array_filter($terms_array);
    // Create multilevel array.
    return $this->buildNamesTree($terms_array);
  }

  /**
   * Sanitizes term name to be safe.
   *
   * @param string $name
   *  Original term name.
   *
   * @return string
   *  Sanitized term name.
   */
  private function sanitizeName(string $name): string {
    $name = htmlspecialchars(strip_tags($name), ENT_QUOTES, 'UTF-8');
    return Html::escape($name);
  }

  /**
   * Retrieve max length for the name field.
   *
   * @param string $vid
   *   Current vocabulary id.
   *
   * @return int
   *   Field max length.
   */
  private function getVocabularyMaxLength(string $vid): int {
    // Default max width of text field.
    $max_length = 255;
    $definitions = \Drupal::service('entity_field.manager')->getFieldDefinitions('taxonomy_term', $vid);
    if (!empty($definitions['name'])) {
      $fieldDefinition = $definitions['name'];
      $itemDefinition = $fieldDefinition->getItemDefinition();
      $max_length = $itemDefinition->getSetting('max_length');
    }
    return $max_length;
  }

  /**
   * Code to convert delimited flat array into a multilevel array.
   *
   * @param array $names
   *   Delimited flat array.
   *
   * @return array
   *   Multilevel Array.
   */
  private function buildNamesTree(array $names): array {
    if (empty($names)) {
      return [];
    }
    $tree = [];
    $stack = [];
    foreach ($names as $line) {
      $level = strspn($line, '-');
      $name = ltrim($line, '-');
      // Create temp node.
      $node = [
        'name' => $name,
        'children' => []
      ];

      // Root level.
      if ($level === 0) {
        $tree[] = $node;
        $stack[0] = &$tree[count($tree) - 1];
      }
      else {
        // Add as child of previous level.
        $parent = &$stack[$level - 1]['children'];
        $parent[] = $node;
        $stack[$level] = &$parent[count($parent) - 1];
      }
    }
    return $tree;
  }
}
