<?php

namespace Drupal\better_taxonomy\Form;

use Drupal\better_taxonomy\BetterTaxonomyService;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\taxonomy\Form\OverviewTerms;
use Drupal\taxonomy\VocabularyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class CoreOverviewTerms extends FormBase {

  /**
   * The module handler.
   *
   * @var ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The renderer.
   *
   * @var RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The entity repository.
   *
   * @var EntityRepositoryInterface
   */
  protected EntityRepositoryInterface $entityRepository;

  /**
   * The pager manager.
   *
   * @var PagerManagerInterface
   */
  protected PagerManagerInterface $pagerManager;

  /**
   * The entity type manager.
   *
   * @var BetterTaxonomyService
   */
  protected BetterTaxonomyService $betterTaxonomyService;

  protected OverviewTerms $coreClass;

  /**
   * Constructs an OverviewTerms object.
   *
   * @param ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param RendererInterface $renderer
   *   The renderer.
   * @param EntityRepositoryInterface $entityRepository
   *   The entity repository.
   * @param PagerManagerInterface $pagerManager
   *   Pager manager service.
   * @param BetterTaxonomyService $betterTaxonomyService
   *   The Better taxonomy utils service.
   */
  public function __construct(ModuleHandlerInterface $moduleHandler,
                              EntityTypeManagerInterface $entityTypeManager,
                              RendererInterface $renderer,
                              EntityRepositoryInterface $entityRepository,
                              PagerManagerInterface $pagerManager,
                              BetterTaxonomyService $betterTaxonomyService) {
    $this->moduleHandler = $moduleHandler;
    $this->entityTypeManager = $entityTypeManager;
    $this->renderer = $renderer;
    $this->entityRepository = $entityRepository;
    $this->pagerManager = $pagerManager;
    $this->betterTaxonomyService = $betterTaxonomyService;
    $this->coreClass = new OverviewTerms($moduleHandler, $entityTypeManager, $renderer, $entityRepository, $pagerManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('entity.repository'),
      $container->get('pager.manager'),
      $container->get('better_taxonomy.better_taxonomy_service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'taxonomy_overview_terms';
  }

  /**
   * Form constructor.
   *
   * Simply clones core Taxonomy overview form so we can manipulate it.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   * @param VocabularyInterface|null $taxonomy_vocabulary
   *   The vocabulary to display the overview form for.
   *
   * @return array
   *   The form structure.
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?VocabularyInterface $taxonomy_vocabulary = NULL): array {
    $form = [];
    // Send vocabulary to the form submissions.
    $form['vocabulary'] = [
      '#type' => 'hidden',
      '#value' => $taxonomy_vocabulary->id(),
    ];

    $search_parameter = $this->betterTaxonomyService->CheckSearch();
    if (empty($search_parameter)) {
      $form = $this->coreClass->buildForm($form, $form_state, $taxonomy_vocabulary);
    }
    else {
      $terms = $this->betterTaxonomyService->GetSearchList($search_parameter, $taxonomy_vocabulary->id());
      $form['terms'] = $this->betterTaxonomyService->GetSearchResults($taxonomy_vocabulary->id(), $search_parameter, $terms);
    }
    // Create Filter section.
    $this->createSectionFilter($form, $search_parameter);
    return $form;
  }

  /**
   * Creates filtering container in form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param string $search_parameter
   *   String to search.
   *
   * @return void
   */
  private function createSectionFilter(array &$form, string $search_parameter = ''): void {
    $form['filter'] = [
      '#type' => 'container',
      '#weight' => -1,
    ];
    $form['filter']['search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Filter'),
      '#default_value' => $search_parameter,
      "#size" => 30,
      '#attributes' => ['autocomplete' => 'off'],
      '#placeholder' => $this->t('Filter terms by name'),
    ];
    $form['filter']['filter'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
      '#submit' => ['::submitFilter'],
      '#attributes' => [
        'id' => 'submit-filter',
        'class' => ['visually-hidden'],
      ],
    ];
    if (!empty($search_parameter)) {
      $form['filter']['clear_filter'] = [
        '#type' => 'submit',
        '#value' => $this->t('Clear filter'),
        '#submit' => ['::submitClearFilter'],
        '#attributes' => [
          'id' => 'clear-filter'
        ],
      ];
    }
  }

  /**
   * Form submission handler.
   *
   * We just call parent submission.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->coreClass->submitForm($form, $form_state);
  }

  /**
   * Handles custom "Reset to alphabetical" sort.
   *
   * We just call parent submission.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return void
   */
  public function submitReset(array &$form, FormStateInterface $form_state): void {
    $this->coreClass->submitReset($form, $form_state);
  }

  /**
   * Handles custom "Filter" submit.
   *
   * @param array $form
   *    An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *    The current state of the form.
   *
   * @return void
   */
  public function submitFilter(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $options = [
      'taxonomy_vocabulary' => $values['vocabulary'],
    ];
    if (!empty($values['search'])) {
      $options['search'] = $values['search'];
    }
    $redirect_url = Url::fromRoute('better_taxonomy.list', $options);
    $form_state->setRedirectUrl($redirect_url);
  }

  /**
   * Handles custom "Clear filter" submit.
   *
   * @param array $form
   *    An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *    The current state of the form.
   *
   * @return void
   */
  public function submitClearFilter(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $redirect_url = Url::fromRoute('better_taxonomy.list', [
      'taxonomy_vocabulary' => $values['vocabulary'],
    ]);
    $form_state->setRedirectUrl($redirect_url);
  }

}
