<?php

namespace Drupal\better_taxonomy\Form;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\ConfirmFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\taxonomy\VocabularyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class DeleteAllForm extends ConfirmFormBase implements ConfirmFormInterface {

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Vocabulary to delete all terms.
   *
   * @var VocabularyInterface
   */
  protected VocabularyInterface $taxonomy_vocabulary;

  /**
   * Class constructor.
   *
   * @param EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): DeleteAllForm|static {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'better_taxonomy_delete_all_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?VocabularyInterface $taxonomy_vocabulary = NULL): array {
    $this->taxonomy_vocabulary = $taxonomy_vocabulary;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to delete all terms from vocabulary %title?', ['%title' => $this->taxonomy_vocabulary->label()]);
  }

  /**
   * {@inheritdoc}
   *
   * @throws EntityMalformedException
   */
  public function getCancelUrl(): Url {
    return $this->taxonomy_vocabulary->toUrl('overview-form');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This will delete all terms from vocabulary %title. This action cannot be undone.',
      ['%title' => $this->taxonomy_vocabulary->label()]
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Delete all terms');
  }

  /**
   * {@inheritdoc}
   *
   * @throws EntityMalformedException
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $taxonomy_manager = $this->entityTypeManager->getStorage('taxonomy_term');
    $terms = $taxonomy_manager->loadTree($this->taxonomy_vocabulary->id(), 0, NULL, TRUE);
    $deleted = 0;
    $message = $this->t('There are no terms to delete.');
    if (!empty($terms)) {
      foreach ($terms as $term) {
        if ($term) {
          $term->delete();
          $deleted++;
        }
      }
      $message = $this->t('@count term(s) deleted.', ['@count' => $deleted]);
    }
    $this->messenger()->addMessage($message);
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
